import db from "../db-config.js";
import {
  countQueryCondition,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteFile,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  insertNotification,
  makeJoins,
  processUploadDocuments,
  searchConditionRecord,
  uniqueIdGenerator,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import { createQueryBuilder, updateQueryBuilder } from "../helper/queryBuilder.js";
import ContractorIncident from "../sequelize/ContractorIncidentSchema.js";
import ContractorIncidentDocs from "../sequelize/ContractorIncidentDocsSchema.js";

/**Function to create and update ContractorIncident  */
export const createUpdatedContractorIncident = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const { id, description, current_impact, potential_impact, organization, department, sidebar_id = 273 } = req.body;

  req.body.description = await encodeSingle_statement(description);
  req.body.current_impact = await encodeSingle_statement(current_impact);
  req.body.potential_impact = await encodeSingle_statement(potential_impact);

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  const documents = await processUploadDocuments(req, sidebar_id, "photographs_incident");

  if (id && documents.length) {
    const deleteQuery = `UPDATE contractor_incident_docs SET deleted = 1 where deleted = 0 AND contractor_incident_id = ?`;
    await db.query(deleteQuery, [id]);
  }
  for (const i of req.body.notification) {
    await insertNotification(
      "ContractorIncident",
      "An contractor_incident has occurred",
      i,
      "text",
      req.user.sessionid
    );
  }

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  if (!id) {
    const incidentId = await uniqueIdGenerator(
      organizationId,
      department,
      "CIN",
      "contractor_incident",
      "incident_report_id"
    );
    req.body.incident_report_id = incidentId;
  }

  const { query, values } = id
    ? updateQueryBuilder(ContractorIncident, req.body)
    : createQueryBuilder(ContractorIncident, req.body);
  // console.log(query, values, "query, values");
  const [result] = await db.query(query, values);

  if (documents || Array.isArray(documents)) {
    // Iterate through uploaded files
    console.log(documents, "documents");
    for (const file of documents) {
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(ContractorIncidentDocs, {
        doc_no: id_doc,
        contractor_incident_id: id ? id : result.insertId,
        title,
        ddrm_id,
      });

      await db.query(docQuery, docValues);
    }
  }
  /** Insert record for activity log */
  await insertActivityLog(req.user.sessionid, status, "ContractorIncident", id);

  return sendResponse(res, 200, "Record created successfully");
};

/**Function to view all ContractorIncident */
export const viewAllContractorIncident = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_incident",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "contractor_incident.location",
    "contractor_incident.description",
    "contractor_incident.current_impact",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
    "CONCAT(reporter_user.name , ' ' , reporter_user.surname)",
    "CONCAT(internal_user.name , ' ' , internal_user.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_incident.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_incident.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_incident.department",
    },
    {
      type: "left",
      targetTable: "users as reporter_user",
      onCondition: "reporter_user.id = contractor_incident.incident_reporting_by",
    },
    {
      type: "left",
      targetTable: "roles as reporter_role",
      onCondition: "reporter_user.role = reporter_role.id",
    },
    {
      type: "left",
      targetTable: "department as reporter_department",
      onCondition: "reporter_user.department = reporter_department.id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const incidentDataFetchQuery = `SELECT 
    contractor_incident.id,
    contractor_incident.incident_category,
    contractor_incident.incident_report_id,
    contractor_incident.date_of_incident,
    contractor_incident.time_of_incident_reporting,
    contractor_incident.time_of_incident_occurrence,
    contractor_incident.initial_incident_severity,
    contractor_incident.location,
    contractor_incident.description,
    contractor_incident.current_impact,
    contractor_incident.potential_impact,
    contractor_incident.person_type,
    contractor_incident.internal_persons,
    contractor_incident.external_person,
    contractor_incident.incident_reporting_by,
    contractor_incident.notification,
    contractor_incident.organization,
    contractor_incident.department,
    contractor_incident.created_by,
    contractor_incident.incident_title,
    contractor_incident.is_contractor,
    contractor_incident.contractor,
    reporter_user.role,
    reporter_role.name as reporting_role_name,
    reporter_user.department,
    reporter_department.name as reporting_department,
    CONCAT(reporter_user.name , ' ' , reporter_user.surname) as incident_reporting_by_name,
    reporter_user.profile as incident_reporting_by_profile,
    CONCAT(users.name , ' ' , users.surname) as created_by_name,
    organization.name as organization_name,
    department.name as department_name
    FROM contractor_incident  
    ${joinsRecord} 
    where contractor_incident.deleted = 0 ${searchCondition} ${condition}`;
  //contractor_incident.incident_reference_no,
  // console.log(incidentDataFetchQuery)
  const [incidentDataFetch] = await db.query(incidentDataFetchQuery);
  // return console.log(incidentDataFetch);
  /**Count all ContractorIncident */
  for (const obj of incidentDataFetch) {
    if (obj.person_type == 0 && obj.is_contractor == 0) {
      const [fetch] = await db.query(
        `SELECT CONCAT(users.name , ' ' ,users.surname) AS name , roles.name AS role_name , department.name AS department_name FROM users LEFT JOIN roles ON users.role = roles.id LEFT JOIN department ON users.department = department.id WHERE users.id = ${obj.internal_persons}`
      );
      obj.internal_persons_name = fetch[0]?.name;
      obj.persons_role_name = fetch[0]?.role_name;
      obj.persons_department = fetch[0]?.department_name;
    }
    if (obj.is_contractor == 1) {
      if (obj.contractor) {
        const [contractorDetails] = await db.query(
          `SELECT contractor_name as name FROM contractor_registration WHERE id = ${obj.contractor}`
        );
        obj.contractor_name = contractorDetails[0]?.name;
      }
    }
    // obj.external_person = JSON.parse(obj.external_person || "[]");
    obj.description = await decodeSingle_statement(obj.description);
    obj.current_impact = await decodeSingle_statement(obj.current_impact);
    obj.potential_impact = await decodeSingle_statement(obj.potential_impact);
    const id = obj.id;
    obj.incident_category = JSON.parse(obj?.incident_category || "[]");
    if (obj.incident_category && obj.incident_category.length > 0) {
      const [incidentCategoryList] = await db.query(
        `SELECT name from incident_category WHERE id In (${obj.incident_category}) AND deleted = 0`
      );

      obj.incident_category_name = incidentCategoryList.map((item) => item.name);
    }
    obj.notification = JSON.parse(obj.notification);
    let usersList = [];
    if (obj.notification && obj.notification.length > 0) {
      [usersList] = await db.query(
        `SELECT CONCAT(users.name , ' ' , users.surname) as name , profile ,roles.id as role_id, roles.name as reporting_role_name , 
        department.id as department_id, department.name as reporting_department from users
        LEFT JOIN roles ON roles.id = users.role 
        LEFT JOIN department ON department.id = users.department WHERE users.id In (${obj.notification}) AND users.deleted = "0"`
      );
    }
    // console.log(usersList, "usersList")
    obj.notification_user = usersList;

    const [files] = await db.query(
      `SELECT contractor_incident_docs.*, repository.url as file, document_creation.document_name as title FROM contractor_incident_docs LEFT JOIN repository ON repository.id = contractor_incident_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE contractor_incident_docs.deleted = 0 AND contractor_incident_id = ${id}`
    );

    obj.photographs_incident = files;
  }

  const totalRecord = await countQueryCondition(incidentDataFetchQuery);
  return sendResponse(res, 200, incidentDataFetch, totalRecord);
};

/**Function to delete a specific ContractorIncident */
export const deleteContractorIncident = async (req, res) => {
  const { id } = req.params;
  const { doc_id } = req.body; //Array will be passed of id's of images to delete
  if (doc_id) {
    for (const doc of doc_id) {
      const [fetchDocs] = await db.query(
        "SELECT file FROM contractor_incident_docs where id = ? AND contractor_incident_id = ?",
        [doc, id]
      );
      const fileName = fetchDocs[0].file;
      deleteFile(fileName);
      await db.query("UPDATE contractor_incident_docs SET deleted = 1 WHERE id = ? AND contractor_incident_id = ?", [
        doc,
        id,
      ]);
    }
  }
  // for()
  const deleteRecordQuery = await deleteRecord("contractor_incident", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "ContractorIncident", id);
    return sendResponse(res, 200, "ContractorIncident report record deleted successfully");
  }
};

// const incidentCategoryConstant = {
//   Safety: [
//     { name: "Injury On Duty Technical Form", tableName: "injury_on_duty" },
//     {
//       name: "Explosion Bomb Threat Technical Form",
//       tableName: "explosion_bomb_form",
//     },
//     {
//       name: "HAZMAT and Fire Technical Form",
//       tableName: "hazmat_fire_technical_form",
//     },
//   ],
//   Health: [
//     {
//       name: "Occupational Disease Technical Form",
//       tableName: "occupational_disease_form",
//     },
//   ],
//   Environment: [
//     {
//       name: "Environmental ContractorIncident Technical Form",
//       tableName: "environmental_incidents",
//     },
//   ],
//   Security: [
//     {
//       name: "Security ContractorIncidents Technical Form",
//       tableName: "security_incident_form",
//     },
//   ],
//   Assets: [
//     {
//       name: "Vehicle ContractorIncident Technical Form",
//       tableName: "vehicle_incident_form",
//     },
//     {
//       name: "Infrastructure Technical Form",
//       tableName: "infrastructure_technical_form",
//     },
//   ],
//   Quality: [
//     {
//       name: "Non Compliance Technical Form",
//       tableName: "non_compliance_technical_form",
//     },
//   ],
//   "Near Miss": [
//     { name: "Near Miss Reporting", tableName: "near_miss_reporting" },
//   ],
// };

// /**Function to view contractor_incident Audit trails */
// export const viewContractorIncidentAuditTrail = async (req, res) => {
//   const { id } = req.params;
//   const pageSize = req.query.pageSize;
//   const page = req.query.page;
//   const all = req.query.all;

//   const incidentRecord = await getRecord("contractor_incident", "id", id);
//   if (incidentRecord.length === 0)
//     return sendResponse(res, 404, "Report Not Found");
//   const contractor_incident = incidentRecord[0];
//   contractor_incident.incident_category = JSON.parse(contractor_incident?.incident_category || "[]");
//   const incidentCategoryQuery = `SELECT id, name , description FROM incident_category WHERE deleted = 0 AND id IN (${contractor_incident.incident_category})`;
//   const [incidentCategoryList] = await db.query(incidentCategoryQuery);
//   let result = [];
//   for (let i = 0; i < incidentCategoryList.length; i++) {
//     const item = incidentCategoryList[i];
//     if (incidentCategoryConstant[item.name]) {
//       item.description = await decodeSingle_statement(item.description);
//       const incidentCategoryFormList = incidentCategoryConstant[item.name];
//       for (let j = 0; j < incidentCategoryFormList.length; j++) {
//         const e = incidentCategoryFormList[j];
//         /** Check the from is already added for this contractor_incident */
//         let status = "Incomplete";
//         let form_id = "";
//         if (e.tableName) {
//           const FormCheckQuery = `SELECT id, save_type FROM ${e.tableName} WHERE contractor_incident_id = ${id} AND deleted = 0 ORDER BY id DESC`;
//           const [formRecord] = await db.query(FormCheckQuery);
//           if (formRecord.length > 0) {
//             status = "Completed";
//             form_id = formRecord[0].id;
//           } else {
//             form_id = "";
//           }
//           // console.log(formRecord, "formRecord");
//           if (formRecord[0]?.save_type == "draft") status = "Draft";
//         }

//         result.push({
//           ...item,
//           title: e.name,
//           status: status,
//           form_id: form_id,
//           contractor_incident_id: id,
//         });
//       }
//     }
//   }
//   const totalRecord = result.length;
//   let recordList = result;
//   /**Add manual Pagination */
//   if (all === "false") {
//     recordList = result.slice((page - 1) * pageSize, page * pageSize);
//   }

//   return sendResponse(res, 200, recordList, totalRecord);
// };
