import db from "../db-config.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  processUploadDocuments,
  searchConditionRecord,
  uniqueIdGenerator,
  updateArchiveRepoDoc,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorAgreements from "../sequelize/ContractorAgreementsSchema.js";
import ContractorDocs from "../sequelize/ContractorDocsSchema.js";
import ContractorPreSite from "../sequelize/ContractorPreSiteSchema.js";
import ContractorQuestionnaire from "../sequelize/ContractorQuestionnaireSchema.js";
import ContractorRegistration from "../sequelize/ContractorRegistrationSchema.js";
import FinalHandover from "../sequelize/FinalHandoverSchema.js";

export const createUpdateContractorRegistration = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const { department, id, sidebar_id = 254 } = req.body;
  if (department) {
    req.body.organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
  }

  // console.log(req.body.email)
  // console.log(req.body.cipc_registration_number)
  // console.log(req.body.phone_number)

  const [check] = await db.query(
    `SELECT * FROM contractor_registration WHERE (phone_number = '${req.body.phone_number}' OR email = '${req.body.email}' OR cipc_registration_number = '${req.body.cipc_registration_number}') AND deleted = 0`
  );

  // console.log(check)

  if (check.length > 0 && !id) {
    return sendResponse(res, 409, "Contractor already exists");
  }
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  const documents = await processUploadDocuments(req, sidebar_id);

  if (id && documents.length) {
    const deleteQuery = `UPDATE contractor_docs SET deleted = 1 where deleted = 0 AND contractor_id = ?`;
    await db.query(deleteQuery, [id]);
  }

  req.body = await encodeAndStringifyFields(req.body);

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "ContractorRegistration",
      "contractor_registration",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  const { query, values } = id
    ? updateQueryBuilder(ContractorRegistration, req.body)
    : createQueryBuilder(ContractorRegistration, req.body);
  // console.log(query, values, "query, values");
  const [result] = await db.query(query, values);

  if (!id) {
    const prePopulateDataFor = {
      questionnaire: {
        model: ContractorQuestionnaire,
        status: "Not Sent",
        modelName: "ContractorQuestionnaire",
        tableName: "contractor_questionnaire",
      },
      agreements: {
        model: ContractorAgreements,
        status: "Not Sent",
        modelName: "ContractorAgreements",
        tableName: "contractor_agreements",
      },
      preSite: {
        model: ContractorPreSite,
        status: "Not Conducted",
        modelName: "ContractorPreSite",
        tableName: "contractor_pre_site",
      },
      finalHandover: {
        model: FinalHandover,
        status: "Not Conducted",
        modelName: "FinalHandover",
        tableName: "final_handover",
      },
    };

    for (let module in prePopulateDataFor) {
      const { model, status, tableName, modelName } = prePopulateDataFor[module];

      const unique_id = await uniqueIdGenerator(
        req.body.organization,
        department,
        modelName,
        tableName,
        "unique_id",
        "unique_id"
      );
      // req.body.unique_id = unique_id;

      const { query: q, values: v } = createQueryBuilder(model, {
        contractor: result.insertId,
        organization: req.body.organization,
        department: req.body.department,
        status,
        created_by: req.user.sessionid,
        unique_id,
      });
      await db.query(q, v);
    }
  }

  if (documents || Array.isArray(documents)) {
    // Iterate through uploaded files
    console.log(documents, "documents");
    for (const file of documents) {
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(ContractorDocs, {
        doc_no: id_doc,
        contractor_id: id ? id : result.insertId,
        title,
        ddrm_id,
      });

      await db.query(docQuery, docValues);
    }
  }
  await insertActivityLog(req.user.sessionid, status, "Contractor Registration", id);
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

export const getContractorRegistration = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "contractor_registration",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "contractor_registration.contractor_name",
    "contractor_registration.phone_number",
    "contractor_registration.email",
    "contractor_registration.physical_address",
    "contractor_registration.cipc_registration_number",
    "contractor_registration.tax_number",
    "contractor_registration.bbbee_status",
    "contractor_registration.vat_registered",
    "contractor_registration.vat_number",
    "contractor_registration.account_number",
    "contractor_registration.bank_code",
    "contractor_registration.account_type",
    "contractor_registration.payment_terms",
    "contractor_registration.work_service_provided", // TEXT (long)
    "contractor_registration.type_of_work", // TEXT (long)
    "contractor_registration.classification_value",
    "contractor_registration.classification_criticality",
    "contractor_registration.classification_impact", // TEXT (long)
    "contractor_registration.classification_category",
    "contractor_registration.contractor_employees", // TEXT (long)
    "contractor_registration.emergency_contacts", // TEXT (long)
    "contractor_registration.contractor_owner",
    "contractor_registration.contractor_name",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
    "banks.name",
  ];

  const searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_registration.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_registration.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = contractor_registration.department",
    },
    {
      type: "left",
      targetTable: "banks",
      onCondition: "banks.id = contractor_registration.bank",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: "repository.id = contractor_registration.ddrm_id",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const contractorRegistrationQuery = `SELECT contractor_registration.* , CONCAT(users.name , ' ' , users.surname) AS created_by_name , organization.name AS organization_name , department.name AS department_name , banks.name AS bank_name, repository.name AS repository_name, repository.url AS repository_url FROM contractor_registration ${joinCondition} WHERE contractor_registration.deleted = 0 ${searchCondition} ${condition}`;

  let [contractorRegistration] = await db.query(contractorRegistrationQuery);
  for (const contractor of contractorRegistration) {
    const [files] = await db.query(
      `SELECT contractor_docs.*, repository.url as file, document_creation.document_name as title FROM contractor_docs LEFT JOIN repository ON repository.id = contractor_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE contractor_docs.deleted = 0 AND contractor_id = ${contractor.id}`
    );

    contractor.upload_documents = files.length ? files : [{ title: "", ddrm_id: null, file: null }];
  }

  contractorRegistration = await decodeAndParseFields(contractorRegistration);

  contractorRegistration = contractorRegistration.map(async (record) => {
    let type_of_work_name = [];
    if (record?.type_of_work && record?.type_of_work?.length > 0) {
      // const type_of_work_ids = record.type_of_work.map((t) => t.type);
      // [type_of_work_name] = await db.query(`SELECT id,name FROM type_of_service WHERE id IN (${type_of_work_ids})`);
    }
    record.type_of_work_name = record?.type_of_work;
    return record;
  });

  contractorRegistration = await Promise.all(contractorRegistration);
  const totalRecord = await countQueryCondition(contractorRegistrationQuery);
  return sendResponse(res, 200, contractorRegistration, totalRecord);
};

export const deleteContractorRegistration = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(ContractorRegistration, id);
  await deleteRecord("contractor_docs", id, "contractor_id");
  await deleteRecord(ContractorQuestionnaire, id, "contractor");
  await deleteRecord(ContractorAgreements, id, "contractor");
  await deleteRecord(ContractorPreSite, id, "contractor");
  await deleteRecord(FinalHandover, id, "contractor");
  await insertActivityLog(req.user.sessionid, "delete", "Contractor Registration", id);
  return sendResponse(res, 200, "Record deleted successfully");
};
