import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  checkNameInOrganization,
  countQueryCondition,
  dateValidator,
  decodeSingle_statement,
  decodeTheEditorContent,
  deleteFile,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  whereCondition,
  storeError,
  uniqueIdGenerator,
  uploadFile,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
} from "../helper/general.js";
import { ICalAlarm } from "ical-generator";
import PermitLicenseCompliance from "../sequelize/PermitLicenseComplianceSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import PermitLicenseDocs from "../sequelize/PermitLicenseDocsSchema.js"; // for sync purpose only not to be removed
/**Function to create and update Permit License  */
export const createUpdatedPermitLicense = async (req, res) => {
  const { id, title, issue_date, end_date, department, organization } =
    req.body;

  req.body = await encodeAndStringifyFields(req.body);

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  /**Unique id generator */

  if (id) {
    /**Update query for permit license */
    req.body.updated_by = req.user.sessionid;
    const { query, values } = updateQueryBuilder(
      PermitLicenseCompliance,
      req.body
    );
    const [updatePermitLicense] = await db.query(query, values);

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Permit License",
      `This user Update Permit License which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "PLC",
      "permit_license_compliance",
      "unique_id"
    );

    req.body.unique_id = unique_id;
    req.body.created_by = req.user.sessionid;

    const { query, values } = createQueryBuilder(
      PermitLicenseCompliance,
      req.body
    );
    const [insertPermittedLicenseUpdate] = await db.query(query, values);

    /** Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Permit License",
      `This user created a new Permit License named '${title}' for organization ${checkNameInOrganization.organization}`
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all Permit License */
export const viewAllPermitLicense = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "permit_license_compliance",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "permit_license_compliance.name",
    "permit_license_compliance.description",
    "permit_license_compliance.permit_holder_type",
    "department.name",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
    "issuing_authority.name",
    "permit_license_compliance.permit_license_category",
    "CONCAT(u2.name , ' ' , u2.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = permit_license_compliance.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = permit_license_compliance.organization",
    },
    {
      type: "left",
      targetTable: "issuing_authority",
      onCondition:
        "issuing_authority.id = permit_license_compliance.issuing_authority",
    },
    {
      type: "left",
      targetTable: "permit_license",
      onCondition: "permit_license.id = permit_license_compliance.name",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = permit_license_compliance.permit_holder",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "u2.role = roles.id",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "u2.department = department.id",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const permitLicenseDataFetchQuery = `SELECT 
    permit_license_compliance.id,
    permit_license_compliance.unique_id,
    permit_license_compliance.name ,permit_license_compliance.permit_license_category , 
    permit_license_compliance.permit_license_number,
    permit_license_compliance.created_by as created_by_id, 
    permit_license_compliance.description,permit_license_compliance.permit_holder ,
    permit_license_compliance.permit_holder_type,
    permit_license_compliance.issuing_authority,
    organization.name as organization_name, 
    issuing_authority.name as issuing_authority_name , 
    permit_license_compliance.applicable_section_of_legislation , 
    permit_license_compliance.issue_date , 
    u2.name as employee,
    department.name as employee_department,
    u2.profile as employee_profile,
    u2.unique_id as employee_number,
    CONCAT(u2.name , " ",u2.surname) as employee_name,
    roles.name as role_name,
    permit_license_compliance.condition_description, 
    permit_license_compliance.created_at , 
    permit_license_compliance.updated_at , 
    permit_license_compliance.organization,
    permit_license_compliance.effective_date,
    permit_license_compliance.end_date,
    permit_license.name as permit_license_name,
    permit_license.id as permit_license_id
    FROM permit_license_compliance  
    ${joinsRecord} 
    where permit_license_compliance.deleted = "0" ${searchCondition} ${condition}`;
  // return console.log(permitLicenseDataFetchQuery)
  let [permitLicenseDataFetch] = await db.query(permitLicenseDataFetchQuery);
  permitLicenseDataFetch = await decodeAndParseFields(permitLicenseDataFetch);
  /**Count all Permit License */
  for (const obj of permitLicenseDataFetch) {
    for (let i = 0; i < obj.condition_description.length; i++) {
      if (obj.condition_description[i].responsiblePerson) {
        let [name] = await db.query(
          `SELECT name,profile FROM users WHERE id = ${obj.condition_description[i].responsiblePerson}`
        );
        obj.condition_description[i].responsible_person_name = name[0]?.name;
        obj.condition_description[i].responsible_person_profile =
          name[0]?.profile;
      }
    }
  }

  const totalRecord = await countQueryCondition(permitLicenseDataFetchQuery);
  return sendResponse(res, 200, permitLicenseDataFetch, totalRecord);
};

/**Function to delete a specific Permit License */
export const deletePermitLicense = async (req, res) => {
  const { id } = req.params;
  const [filePaths] = await db.query(
    "SELECT * FROM permit_license_docs WHERE permit_license_id = ?",
    [id]
  );
  if (filePaths.length > 0) {
    for (const filePath of filePaths) {
      if (filePath.file) {
        await deleteFile(filePath.file);
      }
    }
  }
  await deleteRecord("permit_license_docs", id, "permit_license_id");
  const deleteRecordQuery = await deleteRecord("permit_license_compliance", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Permit License", id);
    return sendResponse(
      res,
      200,
      "Record Deleted successfully"
    );
  }
};
