import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  insertActivityLog,
  searchConditionRecord,
  makeJoins,
  whereCondition,
  countQueryCondition,
  deleteSettingRecord,
  encodeSingle_statement,
  decodeSingle_statement,
  settingsUpdater,
  updateQueryBuilder,
  validateParentId,
  deleteRecord,
  createQueryBuilder,
  uniqueIdGenerator,
} from "../helper/general.js";
import PhysicalLocation from "../sequelize/PhysicalLocationSchema.js";

/** Function to create or update PhysicalLocation */
export const createUpdatePhysicalLocation = async (req, res) => {
  const { id, name, description, parent_id } = req.body;
  const encodedDescription = await encodeSingle_statement(description);

  if (id) {
    if (parent_id == id) {
      return sendResponse(res, 400, `Parent and Child cannot be same`);
    }
    if (parent_id) {
      await validateParentId(id, parent_id);
    }
    req.body.updated_by = req.user.sessionid;
    const { query, values } = updateQueryBuilder(PhysicalLocation, req.body);
    const [result] = await db.query(query, values);
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "PhysicalLocation",
      `This user updated PhysicalLocation with id: ${id}`
    );
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const [checkName] = await db.query(
      `SELECT name FROM physical_location WHERE name = ? AND deleted = 0 AND organization = ?`,
      [name, req.body.organization]
    );
    if (checkName.length === 0) {
      req.body.created_by = req.user.sessionid;

      const unique_id = await uniqueIdGenerator(
        req.body.organization,
        req?.body?.department || null,
        "PhysicalLocation",
        "physical_location",
        "unique_id",
        "unique_id"
      );
      req.body.unique_id = unique_id;

      const { query, values } = createQueryBuilder(PhysicalLocation, req.body);
      const [result] = await db.query(query, values);
      await insertActivityLog(
        req.user.sessionid,
        "create",
        "PhysicalLocation",
        `This user created a new PhysicalLocation '${name}' for organization ${req.body.organization}`
      );
      return sendResponse(res, 200, "Record created successfully");
    } else {
      return sendResponse(res, 400, "PhysicalLocation already exists");
    }
  }
};

/** Function to view all or single PhysicalLocation */
export const viewPhysicalLocation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "physical_location",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchFields = [
    "physical_location.name",
    "physical_location.description",
    "users.name",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchFields
  );
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = physical_location.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = physical_location.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const fetchQuery = `SELECT physical_location.*, CONCAT(users.name, ' ', users.surname) as created_by, organization.name AS organization_name FROM physical_location ${joinsRecord} WHERE physical_location.deleted = 0 ${searchCondition} ${condition}`;
  let [records] = await db.query(fetchQuery);

  for (const record of records) {
    record.description = await decodeSingle_statement(record.description);
    //  if (req.query.grouped == "true") {
    //   const [organizations] = await db.query(
    //     `SELECT organization FROM physical_location WHERE deleted = 0 AND name = ?`,
    //     [record.name]
    //   );
    //   const arr = organizations.map((item) => item.organization);
    //   record.organizations = arr;
    // }
  }

  const totalRecord = await countQueryCondition(fetchQuery);
  return sendResponse(res, 200, records, totalRecord);
};

/** Function to delete PhysicalLocation */
export const deletePhysicalLocation = async (req, res) => {
  const { id } = req.params;
  // const deleteRecord = await deleteSettingRecord("physical_location", id);
  const deleteRecord = await deleteRecord("physical_location", id);
  if (deleteRecord) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "PhysicalLocation",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  } else {
    return sendResponse(res, 404, "Record not found");
  }
};
