import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  countQueryCondition,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  whereCondition,
  encodeSingle_statement,
  encodeTheEditorContent,
  decodeSingle_statement,
  decodeTheEditorContent,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  decodeAndParseFields,
  riskRegisterTemplateHelper,
} from "../helper/general.js";
import StrategicRisk from "../sequelize/StrategicRiskSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import StrategicRiskIdentification from "../sequelize/StrategicRiskIdentificationSchema.js";
import { getStrategicRiskAndItsIdentification } from "./strategicRiskController.js";

/**Function to create new Risk identification  */
export const strategicRiskIdentificationCreateUpdate = async (req, res) => {
  const {
    id,
    risk_description,
    impact_description,
    causes_description,
    control_description,
    opportunity_description,
    organization,
    department,
  } = req.body;

  req.body.causes_description = await encodeSingle_statement(
    causes_description
  );
  req.body.control_description = await encodeSingle_statement(
    control_description
  );
  req.body.impact_description = await encodeSingle_statement(
    impact_description
  );
  req.body.risk_description = await encodeSingle_statement(risk_description);
  // req.body.opportunity_identification = req.body.opportunity_identification == "1" ? 1 : 0;
  req.body.opportunity_description =
    opportunity_description.length > 0
      ? JSON.stringify(
          await encodeTheEditorContent(opportunity_description, "description")
        )
      : "[{}]";

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  /**If id comes in body then it will update the query */
  if (id) {
    delete req.body.updated_at;
    req.body.updated_by = req.user.sessionid;
    /**Update Strategic Risk Identification Query */
    const { query, values } = updateQueryBuilder(
      StrategicRiskIdentification,
      req.body
    );

    const [updateStrategicRiskIdentification] = await db.query(query, values);

    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "Roles", id);

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    req.body.created_by = req.user.sessionid;
    /**Insert record for Strategic Risk Identification */
    const { query, values } = createQueryBuilder(
      StrategicRiskIdentification,
      req.body
    );

    const [createStrategicRiskIdentification] = await db.query(query, values);

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Strategic Risk Identification",
      createStrategicRiskIdentification.insertId
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single Strategic Risk Identification */
export const viewStrategicRiskIdentification = async (req, res) => {
  const { id } = req.params;
  const { approval_status } = req.query;
  const approvalFilter = approval_status
    ? `AND strategic_risk.approval_status = "${approval_status}"`
    : "";
  const status = req.query.status || false;

  const condition = await whereCondition({
    table: "strategic_risk_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(riskOwner.name , ' ' , riskOwner.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(control.name , ' ' , control.surname)",
    "strategic_risk.unique_id",
    "strategic_risk.approval_status",
    "strategic_risk.risk_register_name",
    "strategic_risk.risk_context",
    "strategic_risk.status",
    "strategic_risk_identification.causes_description",
    "strategic_risk_identification.impact_description",
    "strategic_risk_identification.risk_description",
    "strategic_risk_identification.inherent_likelihood",
    "strategic_risk_identification.inherent_consequence",
    "strategic_risk_identification.inherent_risk_rating",
    "strategic_risk_identification.inherent_risk_ranking",
    "strategic_risk_identification.residual_risk_rating",
    "strategic_risk_identification.residual_risk_ranking",
    "strategic_risk_identification.opportunity_description",
    "strategic_risk_identification.risk_name",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = strategic_risk_identification.created_by",
    },
    {
      type: "left",
      targetTable: "users as control",
      onCondition: "control.id = strategic_risk_identification.id",
    },
    {
      type: "left",
      targetTable: "users as riskOwner",
      onCondition: "riskOwner.id = strategic_risk_identification.risk_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = strategic_risk_identification.organization",
    },
    {
      type: "left",
      targetTable: "strategic_risk",
      onCondition:
        "strategic_risk.id = strategic_risk_identification.risk_register_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);
  const statusQuery = status ? `AND strategic_risk.status = "${status}"` : "";

  /**Record of all roles */
  // const strategicRiskIdentificationQuery = `SELECT strategic_risk_identification.id , strategic_risk_identification.risk_register_id , strategic_risk_identification.risk_owner , strategic_risk_identification.risk_name , strategic_risk_identification.risk_description , strategic_risk_identification.impact_description , strategic_risk_identification.causes_description , strategic_risk_identification.inherent_likelihood , strategic_risk_identification.inherent_consequence , strategic_risk_identification.inherent_risk_rating , strategic_risk_identification.inherent_risk_ranking , strategic_risk_identification.control_id , strategic_risk_identification.control_name , strategic_risk_identification.control_description , strategic_risk_identification.control_design_intent , strategic_risk_identification.control_effectiveness , strategic_risk_identification.control_identification , strategic_risk_identification.residual_risk_rating , strategic_risk_identification.residual_risk_ranking , strategic_risk_identification.opportunity_identification , strategic_risk_identification.opportunity_description , strategic_risk_identification.priority_confirmation , strategic_risk_identification.organization , strategic_risk_identification.department,strategic_risk_identification.created_by as created_by_id ,users.name as created_by,users.surname as created_by_surname,users.profile as created_by_profile , strategic_risk .risk_register_name , strategic_risk.approval_status , strategic_risk.status , control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name
  //   FROM strategic_risk_identification
  //   ${joinsRecord} where strategic_risk_identification.deleted = 0  ${searchCondition} ${condition}`;

  const strategicRiskIdentificationQuery = `SELECT strategic_risk_identification.id , strategic_risk_identification.risk_register_id , strategic_risk_identification.risk_owner , strategic_risk_identification.risk_name , strategic_risk_identification.risk_description , strategic_risk_identification.impact_description , strategic_risk_identification.causes_description , strategic_risk_identification.control_identification , strategic_risk_identification.residual_risk_rating , strategic_risk_identification.residual_risk_ranking ,  strategic_risk_identification.inherent_likelihood , strategic_risk_identification.inherent_consequence , strategic_risk_identification.inherent_risk_rating , strategic_risk_identification.inherent_risk_ranking , strategic_risk_identification.control_id , strategic_risk_identification.opportunity_identification , strategic_risk_identification.opportunity_description , strategic_risk_identification.priority_confirmation , strategic_risk_identification.organization , strategic_risk_identification.department,strategic_risk_identification.created_by as created_by_id ,users.name as created_by,users.surname as created_by_surname,users.profile as created_by_profile , strategic_risk.risk_register_name , strategic_risk.unique_id, strategic_risk.approval_status , strategic_risk.status , control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name
    FROM strategic_risk_identification 
    ${joinsRecord} where strategic_risk_identification.deleted = 0 ${statusQuery} ${approvalFilter}  ${searchCondition} ${condition}`;

  // Modify the ORDER BY clause to ASC
  let query = strategicRiskIdentificationQuery.replace(
    /ORDER BY .*? DESC/,
    "ORDER BY strategic_risk_identification.residual_risk_ranking ASC"
  );
  let [strategicRiskIdentificationDataFetch] = await db.query(query);
  // console.log(strategicRiskIdentificationQuery);

  strategicRiskIdentificationDataFetch = await decodeAndParseFields(
    strategicRiskIdentificationDataFetch
  );
  // console.log(roles);

  const totalRecord = await countQueryCondition(
    strategicRiskIdentificationQuery
  );

  return sendResponse(
    res,
    200,
    strategicRiskIdentificationDataFetch,
    totalRecord
  );
};

export const updateStrategicRiskIdentificationPriority = async (req, res) => {
  const { id, priority_confirmation } = req.body;

  if (!id || !priority_confirmation) {
    return sendResponse(
      res,
      400,
      "Please provide id and priority_confirmation"
    );
  }

  let [riskIdentification] = await db.query(
    `SELECT risk_name, risk_register_id, risk_description, CONCAT(users.name , ' ' , users.surname) as risk_owner_name, users.email as risk_owner_email FROM strategic_risk_identification LEFT JOIN users ON users.id = strategic_risk_identification.risk_owner WHERE strategic_risk_identification.id = ? AND strategic_risk_identification.deleted = 0`,
    [id]
  );

  riskIdentification = await decodeAndParseFields(riskIdentification);

  const additionalData = {
    risk_name: riskIdentification[0].risk_name,
    risk_description: riskIdentification[0].risk_description,
    risk_owner_name: riskIdentification[0].risk_owner_name,
  };

  const { query, values } = updateQueryBuilder(
    StrategicRiskIdentification,
    req.body
  );
  const [updatePriority] = await db.query(query, values);

  if (updatePriority.affectedRows > 0) {
    const templateFileUrl = "mail_for_risk_prioritised_template.html";
    const templateName = "Risk Prioritised";
    const subject = `Priority Risk Notification - ${riskIdentification[0].risk_name}`;
    await riskRegisterTemplateHelper(
      templateFileUrl,
      templateName,
      subject,
      riskIdentification[0].risk_register_id,
      true,
      riskIdentification[0].risk_owner_email,
      additionalData
    );
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    return sendResponse(res, 400, "Record not updated");
  }
};

/**Function to view all and single Strategic Risk Identification */
export const viewStrategicRiskIdentificationApprovalWorkflow = async (
  req,
  res
) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "strategic_risk_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(riskOwner.name , ' ' , riskOwner.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(control.name , ' ' , control.surname)",
    "strategic_risk.unique_id",
    "strategic_risk.approval_status",
    "strategic_risk.risk_register_name",
    "strategic_risk.risk_context",
    "strategic_risk.status",
    "strategic_risk_identification.causes_description",
    "strategic_risk_identification.impact_description",
    "strategic_risk_identification.risk_description",
    "strategic_risk_identification.inherent_likelihood",
    "strategic_risk_identification.inherent_consequence",
    "strategic_risk_identification.inherent_risk_rating",
    "strategic_risk_identification.inherent_risk_ranking",
    "strategic_risk_identification.residual_risk_rating",
    "strategic_risk_identification.residual_risk_ranking",
    "strategic_risk_identification.opportunity_description",
    "strategic_risk_identification.risk_name",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = strategic_risk_identification.created_by",
    },
    {
      type: "left",
      targetTable: "users as control",
      onCondition: "control.id = strategic_risk_identification.id",
    },
    {
      type: "left",
      targetTable: "users as riskOwner",
      onCondition: "riskOwner.id = strategic_risk_identification.risk_owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = strategic_risk_identification.organization",
    },
    {
      type: "left",
      targetTable: "strategic_risk",
      onCondition:
        "strategic_risk.id = strategic_risk_identification.risk_register_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all roles */
  // const strategicRiskIdentificationQuery = `SELECT strategic_risk_identification.id , strategic_risk_identification.risk_register_id , strategic_risk_identification.risk_owner , strategic_risk_identification.risk_name , strategic_risk_identification.risk_description , strategic_risk_identification.impact_description , strategic_risk_identification.causes_description , strategic_risk_identification.inherent_likelihood , strategic_risk_identification.inherent_consequence , strategic_risk_identification.inherent_risk_rating , strategic_risk_identification.inherent_risk_ranking , strategic_risk_identification.control_id , strategic_risk_identification.control_name , strategic_risk_identification.control_description , strategic_risk_identification.control_design_intent , strategic_risk_identification.control_effectiveness , strategic_risk_identification.control_owner , strategic_risk_identification.residual_risk_rating , strategic_risk_identification.residual_risk_ranking , strategic_risk_identification.opportunity_identification , strategic_risk_identification.opportunity_description , strategic_risk_identification.priority_confirmation , strategic_risk_identification.organization , strategic_risk_identification.department,strategic_risk_identification.created_by as created_by_id ,users.name as created_by,users.surname as created_by_surname,users.profile as created_by_profile , strategic_risk .risk_register_name , strategic_risk.approval_status , strategic_risk.status , control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name
  //   FROM strategic_risk_identification
  //   ${joinsRecord} where strategic_risk_identification.deleted = 0 AND strategic_risk.status = 'Complete' AND (strategic_risk.approval_status != 'Approved'  AND  strategic_risk.approval_status != 'Rejected') ${searchCondition} ${condition}`;

  const strategicRiskIdentificationQuery = `SELECT strategic_risk_identification.id , strategic_risk_identification.risk_register_id , strategic_risk_identification.risk_owner , strategic_risk_identification.risk_name , strategic_risk_identification.risk_description , strategic_risk_identification.impact_description , strategic_risk_identification.causes_description , strategic_risk_identification.inherent_likelihood , strategic_risk_identification.inherent_consequence , strategic_risk_identification.inherent_risk_rating , strategic_risk_identification.inherent_risk_ranking , strategic_risk_identification.control_id , strategic_risk_identification.control_identification , strategic_risk_identification.residual_risk_rating , strategic_risk_identification.residual_risk_ranking , strategic_risk_identification.opportunity_identification , strategic_risk_identification.opportunity_description , strategic_risk_identification.priority_confirmation , strategic_risk_identification.organization , strategic_risk_identification.department,strategic_risk_identification.created_by as created_by_id ,users.name as created_by,users.surname as created_by_surname,users.profile as created_by_profile , strategic_risk.risk_register_name , strategic_risk.unique_id, strategic_risk.approval_status , strategic_risk.status , control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , organization.name as organization_name
    FROM strategic_risk_identification 
    ${joinsRecord} where strategic_risk_identification.deleted = 0 AND strategic_risk.status = 'Complete' AND (strategic_risk.approval_status != 'Approved'  AND  strategic_risk.approval_status != 'Rejected') ${searchCondition} ${condition}`;
  let [strategicRiskIdentificationDataFetch] = await db.query(
    strategicRiskIdentificationQuery
  );
  // console.log(strategicRiskIdentificationQuery);

  strategicRiskIdentificationDataFetch = await decodeAndParseFields(
    strategicRiskIdentificationDataFetch
  );
  // console.log(roles);

  const totalRecord = await countQueryCondition(
    strategicRiskIdentificationQuery
  );

  return sendResponse(
    res,
    200,
    strategicRiskIdentificationDataFetch,
    totalRecord
  );
};

/**Function to delete a specific Strategic Risk Identification */
export const deleteStrategicRiskIdentification = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord(
    "strategic_risk_identification",
    id
  );
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Strategic Risk Identification",
      id
    );
    return sendResponse(res, 200, "Record Deleted successfully");
  }
};

/**Function to delete a specific Strategic Risk Identification */
export const approveRejectStrategicRiskIdentification = async (req, res) => {
  const { strategicRiskIdentificationId, status, comment } = req.body;
  const updateStrategicRiskIdentificationQuery = `UPDATE strategic_risk SET approval_status=?, reject_comment=?, updated_by=? WHERE id= ${strategicRiskIdentificationId};`;
  const updateStrategicRiskIdentificationValues = [
    status,
    comment,
    req.user.sessionid,
  ];
  const [updateStrategicRiskIdentification] = await db.query(
    updateStrategicRiskIdentificationQuery,
    updateStrategicRiskIdentificationValues
  );

  /**Insert record for activity log */
  await insertActivityLog(
    req.user.sessionid,
    "approved",
    "Strategic Risk Identification",
    strategicRiskIdentificationId
  );
  return sendResponse(res, 200, `Record ${status} successfully`);
};
