import db from "../db-config.js";
import {
  insertActivityLog,
  storeError,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  dateValidator,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  encodeAndStringifyFields,
  decodeAndParseFields,
  getRecord,
  insertNotification,
  uniqueIdGenerator,
  generateStrategicTacticalTables,
  sendDynamicEmail,
  riskRegisterTemplateHelper,
} from "../helper/general.js";
import TacticalRisk from "../sequelize/TacticalRiskSchema.js";
import TacticalRiskIdentification from "../sequelize/TacticalRiskIdentificationSchema.js";
import { sendResponse } from "../helper/wrapper.js";

export async function sendTacticalCreationEmail(req, id) {
  try {
    const { strategicRisk } = await getTacticalData(req, id);
    const strategicData = strategicRisk[0];
    const companyWebsite = process.env.HARMONY_FRONTEND_URL;

    const {
      controlIdentificationTable,
      inherentRiskTable,
      opportunityIdentificationTable,
      overallResidualRatingTable,
      residualRatingTable,
      strategicRiskIdentifiedTable,
    } = generateStrategicTacticalTables(strategicData);

    let usersToSend = [
      {
        name: `${strategicData.facilitator_name} ${strategicData.facilitator_surname}`,
        email: strategicData.facilitator_email,
      },
      {
        name: `${strategicData.approver_name} ${strategicData.approver_surname}`,
        email: strategicData.approver_email,
      },
    ];
    strategicData.risk_identification_details.forEach((risk) => {
      usersToSend.push({
        name: `${risk.risk_owner_name} ${risk.risk_owner_surname}`,
        email: risk.risk_owner_email,
      });
    });

    for (let user of usersToSend) {
      const sendRecordArray = {
        name: user.name,
        templateFileUrl: "mail_for_strategic_tactical_register_template.html",
        templateName: "Strategic Risk/ Tactical Risk Register creation",
        company_name: strategicData.organization_name,
        company_website: companyWebsite,
        risk_register_name: strategicData.name,
        risk_assessment_date: moment(strategicData.assessment_date).format(
          "DD-MM-YYYY"
        ),
        status: strategicData.status,
        business_structure: strategicData.organization_name,
        risk_review_date: moment(strategicData.review_date).format(
          "DD-MM-YYYY"
        ),
        risk_assessment_facilitator: `${strategicData.facilitator_name} ${strategicData.facilitator_surname}`,
        risk_register_approver: `${strategicData.approver_name} ${strategicData.approver_surname}`,
        risk_meeting_name: strategicData.meeting,
        strategicRisksIdentified: strategicRiskIdentifiedTable,
        inherentRisk: inherentRiskTable,
        controlIdentification: controlIdentificationTable,
        residualRating: residualRatingTable,
        overallResidualRating: overallResidualRatingTable,
        opportunityIdentification: opportunityIdentificationTable,
      };
      await sendDynamicEmail({
        to: user.email,
        subject: `Successful Creation of Risk Register - ${strategicData.name}`,
        data: sendRecordArray,
      });
    }
    return true;
  } catch (error) {
    console.log("Error in sending risk creation email: ", error);
  }
}

/** Function to create/update Tactical risk */
export const tacticalRiskCreateUpdate = async (req, res) => {
  const {
    id,
    assessment_date,
    review_date,
    organization,
    department,
    risk_identification,
    status,
  } = req.body;

  /** Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }
  req.body.assessment_date = dateValidator(assessment_date);
  req.body.review_date = dateValidator(review_date);

  if (id) {
    req.body.updated_by = req.user.sessionid;
    /** add a check whether tactical risk is rejected before updating */
    const [statusCheck] = await getRecord("tactical_risk", "id", id);
    if (statusCheck.approval_status == "Rejected") {
      return sendResponse(res, 400, "Record already rejected !");
    }
    /** Update TacticalRisk Query */
    const { query, values } = updateQueryBuilder(TacticalRisk, req.body);
    await db.query(query, values);

    if (status === "Complete" && statusCheck.status == "Draft") {
      const templateFileUrl =
        "mail_for_strategic_tactical_register_approval_template.html";
      const templateName = "Strategic Risk/ Tactical Risk Register Approval";
      const subject = `Risk Register Awaiting Your Approval - ${risk_register_name}`;
      await riskRegisterTemplateHelper(
        templateFileUrl,
        templateName,
        subject,
        id,
        false
      );
    }

    if (
      statusCheck.approval_status == "Pending Approval" &&
      req.body.approval_status == "Approved"
    ) {
      const templateFileUrl =
        "mail_for_strategic_tactical_register_approved_template.html";
      const templateName = "Strategic Risk/ Tactical Risk Register Approved";
      const subject = `Risk Register Approval - ${risk_register_name}`;
      await riskRegisterTemplateHelper(
        templateFileUrl,
        templateName,
        subject,
        id,
        false
      );
    }

    /** Also update all risk identification related to this tactical risk */
    for (let i = 0; i < risk_identification.length; i++) {
      let riskIdentification = risk_identification[i];
      riskIdentification.risk_register_id = id;
      riskIdentification.updated_by = req.user.sessionid;
      riskIdentification.organization = req.body.organization;
      riskIdentification.department = req.body.department;
      const riskIdentificationId = riskIdentification.id;
      riskIdentification = await encodeAndStringifyFields(riskIdentification);
      const { query, values } = riskIdentificationId
        ? updateQueryBuilder(TacticalRiskIdentification, riskIdentification)
        : createQueryBuilder(TacticalRiskIdentification, riskIdentification);
      await db.query(query, values);
    }

    if (status == "Draft") {
      await sendTacticalCreationEmail(req, id);
    }

    /** Insert Activity */
    await insertActivityLog(req.user.sessionid, "update", "TacticalRisk", id);

    const [record] = await getRecord("tactical_risk", "id", id);
    /// send a notification to risk owner that strategic risk has been created
    if (status == "Complete" && record.approval_status != "Rejected") {
      const userId = req.body?.risk_register_approver;
      const pageUrl = `/risk/tactical-risk-register?tab=approval-workflow`;
      if (userId) {
        await insertNotification(
          "Tactical Risk",
          pageUrl,
          userId,
          "url",
          req.user.sessionid
        );
      }
    }

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "TAC",
      "tactical_risk",
      "unique_id",
      "unique_id"
    );
    const type = "reference";
    const reference_no = await uniqueIdGenerator(
      organizationId,
      department,
      "TAC",
      "tactical_risk",
      "reference_no",
      type
    );
    req.body.unique_id = unique_id;
    req.body.reference_no = reference_no;
    req.body.created_by = req.user.sessionid;
    /** Insert record for TacticalRisk */
    const { query, values } = createQueryBuilder(TacticalRisk, req.body);
    const [createTacticalRisk] = await db.query(query, values);

    /** Add all risk_identification related to this tactical risk */
    for (let i = 0; i < risk_identification.length; i++) {
      let riskIdentification = risk_identification[i];
      riskIdentification.risk_register_id = createTacticalRisk.insertId;
      riskIdentification.created_by = req.user.sessionid;
      riskIdentification.organization = req.body.organization;
      riskIdentification.department = req.body.department;
      riskIdentification = await encodeAndStringifyFields(riskIdentification);
      const { query, values } = createQueryBuilder(
        TacticalRiskIdentification,
        riskIdentification
      );
      await db.query(query, values);
    }

    await sendTacticalCreationEmail(req, createTacticalRisk.insertId);

    /** Insert Activity */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "TacticalRisk",
      createTacticalRisk.insertId
    );

    const [record] = await getRecord(
      "tactical_risk",
      "id",
      createTacticalRisk.insertId
    );
    /// send a notification to risk owner that Tactical risk has been created
    if (status == "Complete" && record.approval_status != "Rejected") {
      const userId = req.body?.risk_register_approver;
      const pageUrl = `/risk/tactical-risk-register?tab=approval-workflow`;
      if (userId) {
        await insertNotification(
          "Tactical Risk",
          pageUrl,
          userId,
          "url",
          req.user.sessionid
        );
      }
    }

    return sendResponse(res, 200, "Record created successfully");
  }
};

async function getTacticalData(req, id) {
  const condition = await whereCondition({
    table: "tactical_risk",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(facilitator.name , ' ' , facilitator.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(approver.name , ' ' , approver.surname)",
    "tactical_risk.risk_register_name",
    "tactical_risk.risk_context",
    "tactical_risk.status",
    "tactical_risk.unique_id",
    "tactical_risk.reference_no",
    "meeting.meeting_title",
    "organization.name",
  ];
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = tactical_risk.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = tactical_risk.organization",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = tactical_risk.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as facilitator",
      onCondition: "facilitator.id = tactical_risk.risk_assessment_facilitator",
    },
    {
      type: "left",
      targetTable: "users as approver",
      onCondition: "approver.id = tactical_risk.risk_register_approver",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  const tacticalRiskQuery = `SELECT
   tactical_risk.unique_id,
   tactical_risk.reference_no, 
  tactical_risk.reject_comment,tactical_risk.approval_status ,  tactical_risk.id, tactical_risk.created_by as created_by_id, tactical_risk.organization, organization.name as organization_name, users.name as created_by, users.surname as created_by_surname, users.profile as created_by_profile, tactical_risk.risk_register_name, tactical_risk.assessment_date, tactical_risk.review_date, tactical_risk.risk_context, tactical_risk.risk_assessment_facilitator, tactical_risk.register_assessment_facilitator, tactical_risk.risk_register_approver, tactical_risk.meeting_id, tactical_risk.status, meeting.meeting_title as meeting, facilitator.name as facilitator_name, facilitator.surname as facilitator_surname, facilitator.profile as facilitator_profile,facilitator.email as facilitator_email, approver.name as approver_name, approver.surname as approver_surname, approver.profile as approver_profile, approver.email as approver_email
      FROM tactical_risk 
      ${joinsRecord} WHERE tactical_risk.deleted = 0 ${searchCondition} ${condition}`;
  let [tacticalRisk] = await db.query(tacticalRiskQuery);

  tacticalRisk = await decodeAndParseFields(tacticalRisk);

  for (let risk of tacticalRisk) {
    const id = risk?.id;
    let [riskIdentificationData] = await db.query(
      `SELECT tactical_risk_identification.*, users.name as created_by , tactical_risk .risk_register_name  ,control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile ,riskOwner.email as risk_owner_email, organization.name as organization_name ,users.surname as created_by_surname,users.profile as created_by_profile
      FROM tactical_risk_identification 
     LEFT JOIN users ON users.id = tactical_risk_identification.created_by LEFT JOIN users as control ON control.id = tactical_risk_identification.control_owner LEFT JOIN users as riskOwner ON riskOwner.id = tactical_risk_identification.risk_owner LEFT JOIN organization ON organization.id = tactical_risk_identification.organization LEFT JOIN tactical_risk ON tactical_risk.id = tactical_risk_identification.risk_register_id
      WHERE risk_register_id = '${id}' AND tactical_risk_identification.deleted = 0`
    );
    risk.risk_identification_details = await decodeAndParseFields(
      riskIdentificationData
    );
  }
  return { tacticalRisk, tacticalRiskQuery };
}

/** Function to view all and single Tactical Risk */
export const viewTacticalRisk = async (req, res) => {
  const { id } = req.params;

  const { tacticalRisk, tacticalRiskQuery } = await getTacticalData(req, id);
  const totalRecord = await countQueryCondition(tacticalRiskQuery);

  return sendResponse(res, 200, tacticalRisk, totalRecord);
};

/** Function to delete a specific Tactical Risk */
export const deleteTacticalRisk = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("tactical_risk", id);
  // also deleted identification
  await deleteRecord("tactical_risk_identification", id, "risk_register_id");
  if (deleteRecordQuery) {
    await insertActivityLog(req.user.sessionid, "delete", "TacticalRisk", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Record not found");
};
