import db from "../db-config.js";
import {
  countQueryCondition,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  insertNotification,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import TrainingComments from "../sequelize/TrainingNeedAnalysisCommentsSchema.js"; // for sync purpose only not to be removed
import TrainingNeedAnalysis from "../sequelize/TrainingNeedAnalysisSchema.js";

export const createUpdateTrainingNeedAnalysis = async (req, res) => {
  const { id, save_type ,  organization,
    department, } = req.body;
  let status = id ? "Updated" : "Created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  let arr = [];

  const created_user = req.user.sessionid;

  if (save_type == "draft") {
    req.body.for_approval = 0;
    req.body.approval_status = "Draft";
  } else {
    if (created_user == 1) {
      req.body.for_approval = 1;
      req.body.approval_status = "Approved";
    } else {
      const [teams] = await db.query(`SELECT * FROM teams WHERE deleted = 0`);
      for (const team of teams) {
        let teamMembers = JSON.parse(team.team_members);
        // console.log(teamMembers);
        // console.log(team.team_leader);
        if (teamMembers.includes(created_user)) {
          arr.push(team.team_leader);
          break;
        }
      }
      if (arr.length == 0 || arr[0] == null) {
        req.body.for_approval = 1;
        req.body.approval_status = "Approved";
      } else {
        req.body.send_to_employee = arr;
        req.body.for_approval = 1;
      }
    }
  }

   let organizationId = organization;
    if (department) {
      const recordAccordingToOrganization =
        await getOrganizationAccordingToDepartment(department);
      organizationId = recordAccordingToOrganization[0].organization;
      req.body.organization = organizationId;
    }

  if(!id){
     const unique_id = await uniqueIdGenerator(
          organizationId,
          department,
          "TrainingNeedAnalysis",
          "training_need_analysis",
        //   "hse_appointment",
          "unique_id",
          "unique_id"
        );
        req.body.unique_id = unique_id;
  }

  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(TrainingNeedAnalysis, req.body)
    : createQueryBuilder(TrainingNeedAnalysis, req.body);

  const [result] = await db.query(query, values);
  // console.log(arr);
  if (arr.length > 0) {
    for (let i = 0; i < arr.length; i++) {
      const publicUrl = `training-needs-analysis/approval-workflow/view/${
        id ? id : result.insertId
      }`;
      await insertNotification(
        "Training Need Analysis Approval",
        publicUrl,
        arr[i],
        "url",
        req.user.sessionid
      );
    }
  }
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Training Need Analysis",
    id ? id : result.insertId
  );
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

export const getTrainingNeedAnalysis = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "training_need_analysis",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchTableName = [
    "training_need_analysis.training_type",
    "training_need_analysis.training_purpose",
    "training_need_analysis.approval_status",
    "training_need_analysis.training_name",
    "purpose_of_training .name",
    "CONCAT(createdUser.name , ' ' , createdUser.surname)",
  ];
  const filter = req.query.filter && JSON.parse(req.query.filter);

  let ownerCondition = "";
  if (filter?.for_approval == 1) {
    ownerCondition = req.user.isSuperAdmin
      ? ""
      : `AND JSON_CONTAINS(training_need_analysis.send_to_employee, ${req.user.sessionid} )`;
  }

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joinsRecord = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = training_need_analysis.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = training_need_analysis.organization",
    },
    {
      type: "left",
      targetTable: "purpose_of_training",
      onCondition:
        "purpose_of_training.id = training_need_analysis.training_purpose",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = training_need_analysis.department",
    },
  ];

  const joinCondition = await makeJoins(joinsRecord);

  const viewTrainingNeedAnalysisQuery = `SELECT training_need_analysis.* , purpose_of_training .name as training_purpose_name, createdUser.name as created_by_name, createdUser.surname as created_by_surname, organization.name as organization_name, department.name as department_name , createdUser.profile as created_by_profile FROM training_need_analysis ${joinCondition} WHERE training_need_analysis.deleted = 0 ${searchCondition} ${ownerCondition} ${condition}`;

  let [viewTrainingNeedAnalysis] = await db.query(
    viewTrainingNeedAnalysisQuery
  );

  viewTrainingNeedAnalysis = await decodeAndParseFields(
    viewTrainingNeedAnalysis
  );
  for (const element of viewTrainingNeedAnalysis) {
    const training_need = element?.training_need_analysis || [];
    // console.log(training_need);
    const training_need_analysis = [];
    for (const key of training_need) {
      const id = key.user_id;
      // console.log(id);
      let [empDetails] = await db.query(
        `SELECT users.id AS user_id, CONCAT(users.name , ' ' , users.surname) AS name , gender.name AS gender_name, users.profile , users.surname , users.unique_id AS employee_id , users.gender, users.disability ,users.person_with_disabilities, users.race , users.employee_type , users.role AS role_id, roles.name AS role_name FROM users LEFT JOIN roles ON roles.id = users.role LEFT JOIN gender on gender.id = users.gender WHERE users.id = ? AND users.deleted = '0'
          `,
        [id]
      );

      training_need_analysis.push(empDetails[0]);
    }
    element.training_need_analysis = training_need_analysis;
    /** Get all roles if exists */
    let roles = [];
    if (element?.role && element.role?.length > 0) {
      [roles] = await db.query(
        `SELECT roles.id, roles.name  FROM roles WHERE roles.deleted = '0' AND roles.id IN (${element.role})`
      );
    }
    element.roleDetails = roles;
  }

  const totalRecord = await countQueryCondition(viewTrainingNeedAnalysisQuery);

  return sendResponse(res, 200, viewTrainingNeedAnalysis, totalRecord);
};

export const deleteTrainingNeedAnalysis = async (req, res) => {
  const { id } = req.params;
  await deleteRecord("training_need_analysis", id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Training Need Analysis",
    `This user deleted a new Training Need Analysis Record`
  );
  return sendResponse(res, 200, "Record Deleted Successfully");
};

export const approvedRejectTraining = async (req, res) => {
  let { status, id } = req.body;
  req.body.updated_by = req.user.sessionid;
  req.body.approval_status = status;

  let { query, values } = updateQueryBuilder(TrainingNeedAnalysis, req.body);
  // console.log(query);
  await db.query(query, values);
  if (status != "Approved") {
    await db.query(
      `INSERT INTO training_comments (training_id , comments , status , created_by)  VALUES (?, ?, ?, ?)`,
      [id, req.body.comment, status, req.user.sessionid]
    );
  }
  await insertActivityLog(
    req.user.sessionid,
    "modify",
    "Training Need Analysis",
    `This user ${status} the Training Need Analysis with id ${id}`
  );
  await insertNotification(
    `Training Need Analysis ${status}`,
    `User ${req.user.sessionid} ${status} the training with id ${id} `,
    1,
    "text",
    req.user.sessionid
  );
  return sendResponse(res, 200, `Record ${status} Successfully`);
};
