import db, { buildSelectQuery } from "../db-config.js";
import path from "path";
import fs from "fs";
import { DataTypes, Op } from "sequelize";
import he from "he";
import { load } from "cheerio";
import { fileURLToPath } from "url";
import MeetingNotes from "../sequelize/MeetingNotesSchema.js";
import Policy from "../sequelize/PolicySchema.js";
import { parseDocument } from "htmlparser2";
import SurveyQuestions from "../sequelize/SurveyQuestionsSchema.js";
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
import AWS from "aws-sdk";
import { BlobServiceClient } from "@azure/storage-blob";
import mysqldump from "mysqldump";
import dotenv from "dotenv";
import { Storage } from "@google-cloud/storage"
const superadminApiUrl = process.env.SUPER_ADMIN_BACKEND_URL;

dotenv.config();
export const getRecord = async (table, field, value) => {
  try {
    if (!table || !field || !value) {
      throw new Error("Table, field or value is undefined or null");
    }
    const getQuery = `SELECT * FROM ${table} WHERE ${field} = ? AND deleted = "0"`;
    const [record] = await db.query(getQuery, [value]);
    return record;
  } catch (error) {
    console.log(error.message);
    // storeError(error);
    return [];
  }
};

export const getUserDetails = async (id) => {
  /**Fetch all users details according to id */
  try {
    const userDataFetchQuery = `select users.name,users.profile, users.manager , users.is_super_admin,users.my_organization, organization.name as organization_name,users.organization, users.id,users.unique_id,users.unique_id_status, users.surname, users.email, users.phone, users.id_number, users.joining_date, users.end_date, users.gender, users.disability, users.race, users.employee_type, users.highest_qualification, users.name_of_qualification, users.country, users.state, users.city, users.current_address, users.role, users.role_desc, users.level, users.accountability,users.otp,users.deleted,users.created_at,users.created_by,users.updated_at,users.updated_by,cities.name as city_name,states.name as state_name,country.name as country_name,roles.name as role_name,teams.team_members as team_members_id, users.department, users.client_internal_id FROM users
    LEFT JOIN cities on cities.id = users.city
    LEFT JOIN states on states.id = users.state
    LEFT JOIN country on country.id = users.country
    LEFT JOIN roles on roles.id = users.role
    LEFT JOIN teams ON teams.team_leader = users.id
    LEFT JOIN organization ON organization.id = users.my_organization
    where users.deleted = "0" AND users.id = '${id}'`;
    const [userDataFetch] = await db.query(userDataFetchQuery);

    /**Check that this user id is have some team leader or not */
    const findTeamLeaderQuery = `SELECT teams.team_leader,users.name as team_leader_name FROM teams LEFT JOIN users on users.id = teams.team_leader WHERE teams.deleted = "0" AND JSON_CONTAINS(teams.team_members, '${id}')`;
    const [findTeamLeader] = await db.query(findTeamLeaderQuery);
    let teamLeaderName =
      findTeamLeader.length > 0 ? findTeamLeader[0].team_leader_name : "";

    /**Check that user id have any members in their record or not */
    const findTeamMembersQuery = `SELECT id, team_members FROM teams WHERE teams.deleted = "0" AND team_leader = '${id}'`;
    const [findTeamMembers] = await db.query(findTeamMembersQuery);

    let teamMembers = [];
    if (findTeamMembers.length > 0) {
      const teamMemberIDs = findTeamMembers[0].team_members;
      if (teamMemberIDs?.length > 0) {
        const teamMemberDetailsQuery = `SELECT id, name,profile FROM users WHERE id IN (${teamMemberIDs.join(
          ","
        )})`;
        const [teamMemberDetails] = await db.query(teamMemberDetailsQuery);
        if (teamMemberDetails?.length > 0) {
          teamMembers = await teamMemberDetails.map((row) => ({
            id: row.id,
            name: row.name,
            profile: row.profile,
          }));
        }
      }
    }
    const data = userDataFetch[0];

    data["team_members"] = teamMembers;
    data["team_leader"] = teamLeaderName;
    return data;
  } catch (error) {
    storeError(error);
    throw error;
  }
};

export const employeeTree = async (team) => {
  let teamObj = {
    id: team.team_leader,
    name: team.team_leader_name || "demo",
    position: team.role_name,
    children: [],
    profile: team.profile,
  };
  const team_members = JSON.parse(team?.team_members || "[]");
  for (let member of team_members) {
    const newTeamQuery = `SELECT t.id, t.name, t.team_leader, t.team_members, t.organization, t.department,u.name as team_leader_name,u.profile FROM users u JOIN teams t ON u.id = t.team_leader WHERE t.deleted = "0" AND t.team_leader = ${member}`;
    const [allTeamMembers] = await db.query(newTeamQuery);
    if (allTeamMembers.length > 0) {
      const child = await employeeTree(allTeamMembers[0]);
      teamObj.children.push(child);
    } else {
      /** get the employee details and push it in children */
      const employeeQuery = `SELECT id, name , profile FROM users WHERE id = ${member}`;
      const [employee] = await db.query(employeeQuery);
      employee[0].children = [];
      teamObj.children.push(employee[0]);
    }
  }
  return teamObj;
};

/**Function to make string into where in format */
export const inWhereFormat = async (arrayString) => {
  const arr = Array.isArray(arrayString)
    ? arrayString
    : JSON.parse(arrayString);
  if (!Array.isArray(arr)) throw new Error("Input is not a valid array.");
  const data = "(" + arr.join(",") + ")";
  return data;
};

function getCapitalLetters(module) {
  // Check if the module name length is greater than 4
  if (module.length > 4) {
    // Use a regular expression to match capital letters
    let capitalLetters = module.match(/[A-Z]/g);
    return capitalLetters ? capitalLetters.join("") : ""; // Return capital letters as a string
  } else {
    // If the length is not greater than 4, return an empty string or handle as needed
    return module;
  }
}

// H&HG/TNA/2425/001 -> org_name/module/year/id/sno

export const uniqueIdGeneratorForOrganization = async (
  name,
  moduleName,
  table_name,
  key
) => {
  try {
    let id;
    if (name) {
      const initialsArr = name
        .split(" ")
        .map((name) => name.charAt(0).toUpperCase());
      id = initialsArr.join("");
    }
    moduleName = getCapitalLetters(moduleName);
    if (moduleName) {
      id = id + "/" + moduleName;
    }
    const year = getFinancialYear(new Date());
    if (year) {
      id = id + "/" + year;
    }
    const maxIdGetQuery = `SELECT ${key} FROM ${table_name} ORDER BY id DESC LIMIT 1`;
    const [maxId] = await db.query(maxIdGetQuery);
    let lastNumber;
    if (maxId && maxId[0]) {
      lastNumber = maxId[0]?.id + 1;
    } else {
      lastNumber = 1;
    }

    id = id + "/00" + lastNumber;

    return id;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const uniqueIdGenerator = async (
  organizationId,
  departmentId,
  module,
  table_name,
  key,
  type = "unique_id",
  dataId = null
) => {
  try {
    module = getCapitalLetters(module);
    let id = type === "reference" ? "REF:" : "";
    let organizationKey = "organization";

    //  if organization not coming  first organization parent null will be taken

    // const [organization] = await db.query(
    //   `SELECT parent_id AS parent
    //   FROM organization
    //   WHERE parent_id = 0
    //   ORDER BY created_at DESC
    //   LIMIT 1;`
    // );

    // console.log("organization: ", organization);
    // if (organization.length) {
    //   organizationId = organization[0].id;
    // }
    if (organizationId) {
      id = id + `${await getInitials(organizationId, "organization", "name")}`;
      console.log("id: ", id);
    }
    if (departmentId) {
      id = id + "/" + (await getInitials(departmentId, "department", "name"));
    }
    if (module) {
      id = id + "/" + module;
      console.log("module: ", id);
    }
    const year = getFinancialYear(new Date());
    if (year) {
      id = id + "/" + year;
      console.log("getFinancialYear: ", id);
    }
    /** get serial number max from table  */

    if (table_name === "users") {
      organizationKey = "my_organization";
    }

    if (dataId) {
      id = id + "/" + dataId;
    } else {
      const maxIdGetQuery = `SELECT ${key} FROM ${table_name} WHERE  ${key} LIKE '%${id}%' ORDER BY id DESC LIMIT 1`;
      const [maxId] = await db.query(maxIdGetQuery);
      if (maxId[0]?.[key]) {
        let serialNo = maxId[0][key].split(`${id}/`)[1];
        if (isNaN(serialNo)) {
          serialNo = 1;
        }
        // console.log(parseInt(serialNo));
        id = id + "/00" + (parseInt(serialNo) + 1);
      } else {
        id = id + "/" + "001";
      }
    }

    return id;
  } catch (error) {
    storeError(error);
    return;
  }
};

export async function getInitials(id, table_name, key) {
  try {
    const [initials] = await db.query(
      `SELECT * FROM ${table_name} WHERE id = '${id}'`
    );
    let initial = initials[0]?.[key];
    console.log("initial: ", initial);

    /** pick up all the first letters of the name */
    if (!initial) {
      return "";
    }
    const initialsArr = initial
      .split(" ")
      .map((name) => name.charAt(0).toUpperCase());
    return initialsArr.join("");
  } catch (error) {
    storeError(error);
    return;
  }
}

// export function getFinancialYear(date) {
//   // Assuming financial year starts from April 1st and ends on March 31st of the next year
//   const startMonth = 3; // April (months are zero-based)
//   const startDate = new Date(date.getFullYear(), startMonth, 1);
//   const endMonth = 2; // March (months are zero-based)
//   const endDate = new Date(date.getFullYear() + 1, endMonth, 0); // Last day of March of the next year

//   if (date >= startDate && date <= endDate) {
//     // Current date falls within the financial year
//     return (
//       date.getFullYear().toString().slice(2) +
//       (date.getFullYear() + 1).toString().slice(2)
//     );
//   } else {
//     // Current date falls outside the financial year, it means the financial year should be last year to current year
//     const previousStartDate = new Date(date.getFullYear() - 1, startMonth, 1);
//     const previousEndDate = new Date(date.getFullYear(), endMonth, 0);
//     if (date >= previousStartDate && date <= previousEndDate) {
//       return (
//         (date.getFullYear() - 1).toString().slice(2) +
//         date.getFullYear().toString().slice(2)
//       );
//     }
//   }

//   // If current date does not fall within the financial year or the previous financial year, return null
//   return null;
// }

export function getFinancialYear(date) {
  // Convert to UTC to ensure consistency across servers
  const utcDate = new Date(date.toUTCString());
  const year = utcDate.getUTCFullYear();
  const startMonth = 3; // April (zero-based)
  const startDate = new Date(Date.UTC(year, startMonth, 1)); // April 1st UTC

  if (utcDate >= startDate) {
    // Date is on or after April 1st UTC
    return year.toString().slice(2) + (year + 1).toString().slice(2);
  } else {
    // Date is before April 1st UTC
    return (year - 1).toString().slice(2) + year.toString().slice(2);
  }
}

export const settingsUpdater = async (model, organization, body, user) => {
  try {
    body = JSON.parse(JSON.stringify(body));
    let { description } = body;

    // Fetch the original record name based on the ID
    const [recordName] = await db.query(
      `SELECT name FROM ${model.tableName} WHERE id = ${body.id}`
    );
    const getName = await getRecord(
      model.tableName,
      "name",
      recordName[0].name
    );
    const ids = getName.map((e) => e.id);

    if (getName.length === 0) {
      return false;
    }

    // Get all organizations associated with the current record's name
    const getOrganizations = await getRecord(
      model.tableName,
      "name",
      getName[0]?.name
    );
    if (!organization) {
      organization = getOrganizations.map((e) => e.organization);
    }

    for (const org of organization) {
      // Check for existing records in the target organization that don't match the current IDs
      const [nameCheck] = await db.query(
        `SELECT name, id, organization FROM ${model.tableName} WHERE deleted = 0 AND name = ? AND organization = ? AND id NOT IN (${ids})`,
        [body.name, org]
      );

      if (nameCheck.length === 0) {
        body.updated_by = user.sessionid;
        body.organization = org;

        // Update the parent field if applicable
        if (body.parent) {
          const [checkParent] = await db.query(
            `SELECT name FROM ${model.tableName} WHERE id = ?`,
            [body.parent]
          );
          if (checkParent.length > 0) {
            const [newParent] = await db.query(
              `SELECT id FROM ${model.tableName} WHERE name = ? AND organization = ?`,
              [checkParent[0].name, org]
            );
            if (newParent.length > 0) {
              body.parent = newParent[0].id;
            }
          }
        }

        if (model.tableName === "training_disclaimer") {
          for (let org of organization) {
            const [orgName] = await db.query(
              `SELECT name FROM organization WHERE id = ? AND deleted = 0`,
              [org]
            );
            if (orgName.length) {
              description = description.replace(
                /\[Company Name\]/g,
                orgName[0].name
              );
              body.description = await encodeSingle_statement(description);
            }
          }
        }
        const unique_id = await uniqueIdGenerator(
          org,
          "",
          "AuditTemplate",
          "audit_template",
          "unique_id",
          "unique_id"
        );
        body.unique_id = unique_id;
        const { query, values } = updateQueryBuilder(model, body, {
          organization: org,
          name: getName[0]?.name,
        });
        await db.query(query, values);
      }
    }

    // Handle organizations no longer associated with the record's name
    for (const org of getOrganizations) {
      if (organization.includes(org.organization)) continue;
      await db.query(
        `UPDATE ${model.tableName} SET deleted = '1' WHERE deleted = 0 AND organization = ? AND name = ?`,
        [org.organization, getName[0]?.name]
      );
    }

    const getOrgArray = getOrganizations.map((e) => e.organization);
    for (const org of organization) {
      if (getOrgArray.includes(org)) continue;

      const [nameCheck] = await db.query(
        `SELECT name, id, organization FROM ${model.tableName} WHERE deleted = 0 AND name = ? AND organization = ?`,
        [body.name, org]
      );

      if (nameCheck.length === 0) {
        body.created_by = user.sessionid;
        body.organization = org;

        if (body.parent) {
          const [checkParent] = await db.query(
            `SELECT name FROM ${model.tableName} WHERE id = ?`,
            [body.parent]
          );
          if (checkParent.length > 0) {
            const [newParent] = await db.query(
              `SELECT id FROM ${model.tableName} WHERE name = ? AND organization = ?`,
              [checkParent[0].name, org]
            );
            if (newParent.length > 0) {
              body.parent = newParent[0].id;
            }
          }
        }

        const { query, values } = createQueryBuilder(model, body);
        await db.query(query, values);
      }
    }

    return true;
  } catch (error) {
    console.log(error);
    return false;
  }
};

export const settingObj = {
  department: 1,
  responsibility: 1,
  skills: 1,
  license: 1,
  teams: 1,
  appointment_type: 1,
  parent_objective: 1,
  measurement_metric: 1,
  category_type: 1,
  audit_category: 1,
  focus_area: 1,
  audit_template: 1,
  document_type: 1,
  permit_license: 1,
  issuing_authority: 1,
  incident_category: 1,
  insurance_policy_impact: 1,
  name_of_standard: 1,
  template_category: 1,
  labels: 1,
  certificates: 1,
  financial_year: 1,
  assessment_location: 1,
  assessor_method: 1,
  banks: 1,
  role_on_the_project: 1,
  contractor_forms: 1,
  highest_qualification: 1,
  role_hierarchy: 1,
  disability: 1,
  employee_type: 1,
  equipment: 1,
  location: 1,
  first_aid_box_content: 1,
  ppe_type: 1,
  waste_type: 1,
  waste_disposal_method: 1,
  gender: 1,
  language: 1,
  work_location: 1,
  physical_location: 1,
  document_classification: 1,
  best_practice: 1,
  inspection_location: 1,
  audit_type: 1,
  file_classification: 1,
  asset_type: 1,
  asset_management: 1,
  type_of_service: 1,
  currency: 1,
  purpose_of_training: 1,
  engagement_method: 1,
  category: 1,
};

const levenshteinDistance = (a, b) => {
  const dp = Array.from({ length: a.length + 1 }, () =>
    Array(b.length + 1).fill(0)
  );

  for (let i = 0; i <= a.length; i++) dp[i][0] = i;
  for (let j = 0; j <= b.length; j++) dp[0][j] = j;

  for (let i = 1; i <= a.length; i++) {
    for (let j = 1; j <= b.length; j++) {
      const cost = a[i - 1] === b[j - 1] ? 0 : 1;
      dp[i][j] = Math.min(
        dp[i - 1][j] + 1, // Deletion
        dp[i][j - 1] + 1, // Insertion
        dp[i - 1][j - 1] + cost // Substitution
      );
    }
  }

  return dp[a.length][b.length];
};

// Find the closest table names with a threshold

export const deleteAllRecord = async (req, res) => {
  try {
    const { ids, table } = req.body;
    const sessionId = req.user.sessionid;

    // Step 1: Fetch all table names from the database
    const [tables] = await db.query("SHOW TABLES");
    const tableNames = tables.map((tableObj) => Object.values(tableObj)[0]);

    // Step 2: Check if the provided table name exists
    if (!tableNames.includes(table)) {
      // Find closest table names
      const suggestions = tableNames
        .map((t) => ({
          name: t,
          distance: levenshteinDistance(table, t),
        }))
        .sort((a, b) => a.distance - b.distance)
        .slice(0, 2); // Top 2 closest matches

      return res.status(400).json({
        status: false,
        message: `Table '${table}' does not exist. You can use one of the following table below that matches`,
        suggestions: suggestions.map((s) => s.name),
      });
    }
    const [sidebarRecord] = await db.query(
      `SELECT * FROM sidebar WHERE deleted = 0 AND table_name = ?`,
      [table]
    );
    // if (sidebarRecord.length == 0) {           // ! due to dropdown some table can't be present in sidebar due
    //   return res
    //     .status(400)
    //     .json({ status: false, message: "Sidebar doesn't contain table name" });
    // }
    // Proceed with deletion if the table exists
    if (settingObj[table] || sidebarRecord[0]?.path.includes("/settings")) {
      for (const id of ids) {
        // First get the record and delete all records with that name
        const [record] = await db.query(
          `SELECT name FROM ${table} WHERE deleted = 0 AND id = ?`,
          [id]
        );
        if (record[0]) {
          // Delete all records with that name
          await db.query(`UPDATE ${table} SET deleted = 1 WHERE name = ?`, [
            record[0].name,
          ]);
        }
        insertActivityLog(sessionId, "delete", table, id);
      }
    } else {
      for (const id of ids) {
        const deleteRecordQuery = await deleteRecord(
          table,
          id,
          "id",
          sessionId
        );
        if (deleteRecordQuery.status) {
          insertActivityLog(sessionId, "delete", table, id);
        }
      }
    }

    if (table === "organization") {
      await setSuperAdminPermission();
    }

    return res.status(200).json({
      status: true,
      message: "Records deleted successfully",
    });
  } catch (error) {
    storeError(error);
    return res.status(500).json({
      status: false,
      message: error.message,
    });
  }
};

export const insertActivityLog = async (created_by, status, module, id) => {
  try {
    status = status.toLowerCase();
    let message = "";
    if (status == "update") {
      message = `${module} updated successfully with id ${id}`;
    } else if (status == "create") {
      message = `New ${module} created successfully with id ${id}`;
    } else if (status == "delete") {
      message = `${module} deleted successfully with id ${id}`;
    } else if (status == "approved") {
      message = `${module} approved successfully with id ${id}`;
    } else if (status == "reject") {
      message = `${module} reject successfully with id ${id}`;
    } else {
      message = `${module} ${status} successfully with id ${id}`;
    }

    // const message = status == update ? `${module} updated successfully with id ${id}` : `New ${module} updated successfully with id ${id}`
    return await db.query(
      `INSERT INTO activity_log(created_by, status,module, message) VALUES ("${created_by}","${status}","${module}","${message}")`
    );
  } catch (error) {
    storeError(error);
    return;
  }
};

export const checkNameInOrganization = async (
  table,
  name,
  organization,
  field = "name"
) => {
  try {
    const [data] = await db.query(
      `SELECT ${field},id FROM ${table} WHERE ${field} = ? AND deleted = '0' AND organization = ?`,
      [name, organization]
    );
    return data;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const deleteSettingRecord = async (table, id) => {
  try {
    const [getName] = await db.query(
      `SELECT name FROM ${table} WHERE id = ${id}`
    );
    const name = getName[0]?.name;
    const [deleteRecordData] = await db.query(
      `UPDATE ${table} SET deleted='1' WHERE name = '${name}'`
    );
    return deleteRecordData;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const checkNameAlreadyExits = async (
  tableName,
  fieldName,
  fieldData,
  organization
) => {
  try {
    let org = organization ? `AND organization = '${organization}'` : "";
    const query = `select * from ${tableName}  where deleted='0' AND ${fieldName}=? ${org}`;
    const [getData] = await db.query(query, [fieldData]);
    return getData.length > 0;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const insertSurveyQuestions = async (questions, id) => {
  try {
    const [fetch] = await db.query(
      `SELECT * FROM survey_questions WHERE survey_id = ? `,
      [id]
    );
    if (fetch.length > 0) {
      await db.query(`DELETE FROM survey_questions WHERE survey_id = ? `, [id]);
    }
    questions = JSON.parse(questions);
    // console.log(questions);
    for (let question of questions) {
      question.survey_id = id;
      question = await encodeAndStringifyFields(question);
      const { query, values } = createQueryBuilder(SurveyQuestions, question);
      await db.query(query, values);
    }
    return true;
  } catch (error) {
    storeError(error);
    return false;
  }
};

export const recordIdExist = async (table, id) => {
  try {
    if (isNaN(id)) return false;
    const recordIdExistQuery = `SELECT * FROM ${table} WHERE id = ${id} AND deleted = 0`;
    const [recordIdExist] = await db.query(recordIdExistQuery);
    return recordIdExist.length > 0;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const getOrganizationAccordingToDepartment = async (departmentId) => {
  try {
    const [data] = await db.query(
      `SELECT * FROM department WHERE id='${departmentId}' AND deleted = 0`
    );
    return data;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const decodeSingle_statement = async (data) => {
  try {
    if (data) {
      return he.decode(data);
    }
    return null;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const encodeSingle_statement = async (data) => {
  try {
    if (data) {
      return he.encode(data);
    }
    return null;
  } catch (error) {
    storeError(error);
    return;
  }
};
/**Function to print sql query for check */
export const printSqlQuery = async (query, values) => {
  return query.replace(/\?/g, () => {
    const value = values.shift();
    return value !== undefined
      ? typeof value === "string"
        ? `'${value}'`
        : value
      : '""';
  });
};
/**Function to store error in error log file */
export const storeError = async (error) => {
  try {
    console.log(chalk.redBright(error));
    console.log(error);
    const currentTime = new Date().toLocaleString("en-US", {
      timeZone: "Asia/Kolkata",
    });
    const errorMessage = `${currentTime} - Error: ${error.stack}\n`;

    const logDirectory = path.join(__dirname, "../public", "error");
    const logFilePath = path.join(logDirectory, "error.log");

    // Create the directory if it doesn't exist
    if (!fs.existsSync(logDirectory)) {
      fs.mkdirSync(logDirectory, { recursive: true });
    }

    // Append to the error log in the public directory
    fs.appendFileSync(logFilePath, errorMessage);

    // Append to a general error log in the root directory
    const generalLogFilePath = path.join(__dirname, "error.log");
    fs.appendFileSync(generalLogFilePath, errorMessage);
  } catch (err) {
    console.error("Error writing to error log:", err);
  }
};
/**Function to create query builder for insert */
export const createQueryBuilder = (model, requestBody) => {
  if (!model) {
    throw new Error("Model is undefined or null.");
  }
  requestBody = JSON.parse(JSON.stringify(requestBody));
  const columns = Object.keys(model.rawAttributes);
  const filteredColumns = columns.filter((column) => {
    return (
      requestBody.hasOwnProperty(column) &&
      requestBody[column] !== undefined &&
      requestBody[column] !== null &&
      column !== "created_at" &&
      requestBody[column] !== "null" &&
      // requestBody[column] !== "" && // Exclude empty strings
      !(
        model.rawAttributes[column].allowNull === false && !requestBody[column]
      ) && // Check if required field is missing
      !model.rawAttributes[column].primaryKey &&
      !model.rawAttributes[column].autoIncrement
    );
  });
  const placeholders = Array.from(
    { length: filteredColumns.length },
    () => `?`
  ).join(", ");
  const columnNames = filteredColumns.join(", ");
  const query = `INSERT INTO ${model.tableName}(${columnNames}) VALUES (${placeholders})`;
  const values = filteredColumns.map((column) => {
    const attributeType = model.rawAttributes[column].type;
    if (
      (attributeType instanceof DataTypes.JSON ||
        attributeType instanceof DataTypes.TEXT) &&
      (typeof requestBody[column] === "object" ||
        Array.isArray(requestBody[column]))
    ) {
      return JSON.stringify(requestBody[column]);
    }
    if (requestBody[column] === "") {
      return null;
    }
    if (requestBody[column] == null) {
      return null;
    }
    return requestBody[column];
  });
  return {
    query: query,
    values: values,
  };
};

export const isEncoded = (str) => {
  return /&[a-zA-Z0-9#]+;/.test(str);
};

export const decodeAndParseFields = async (dataArray) => {
  return dataArray.map((item) => {
    const decodedItem = { ...item };

    Object.keys(decodedItem).forEach(async (key) => {
      const value = decodedItem[key];

      // Check if the field is a string
      if (typeof value === "string") {
        // Decode encoded fields
        if (isEncoded(value)) {
          decodedItem[key] = await decodeSingle_statement(value);
        }

        // Try to parse stringified JSON
        try {
          const parsedValue = JSON.parse(value);
          decodedItem[key] = parsedValue;
        } catch (e) {
          // Ignore parse errors, keep original value
        }
      }
    });

    return decodedItem;
  });
};

export const decodeAndParseFieldsSync = (dataArray) => {
  return dataArray.map((item) => {
    const decodedItem = { ...item };

    Object.keys(decodedItem).forEach(async (key) => {
      const value = decodedItem[key];

      // Check if the field is a string
      if (typeof value === "string") {
        // Decode encoded fields
        if (isEncoded(value)) {
          decodedItem[key] = await decodeSingle_statement(value);
        }

        // Try to parse stringified JSON
        try {
          const parsedValue = JSON.parse(value);
          decodedItem[key] = parsedValue;
        } catch (e) {
          // Ignore parse errors, keep original value
        }
      }
    });

    return decodedItem;
  });
};

export const encodeAndStringifyFields = async (data) => {
  data = JSON.parse(JSON.stringify(data));
  const dataArray = Array.isArray(data) ? data : [data];
  const processedData = await Promise.all(
    dataArray.map(async (item) => {
      const encodedItem = { ...item };

      await Promise.all(
        Object.keys(encodedItem).map(async (key) => {
          let value = encodedItem[key];

          // Check if the field is a string
          if (typeof value === "string") {
            // Encode the field if it contains HTML
            if (isHtml(value)) {
              encodedItem[key] = await encodeSingle_statement(value);
            }
          }

          // If the value is an object, stringify it
          if (typeof value === "object" && value !== null) {
            encodedItem[key] = JSON.stringify(value);
          }
        })
      );

      return encodedItem;
    })
  );

  return Array.isArray(data) ? processedData : processedData[0];
};

// Helper function to determine if a string contains HTML content
export const isHtml = (str) => {
  const doc = parseDocument(str);
  return doc.childNodes.some((node) => node.type === "tag");
};

/**Function to make query builder for update*/
export const updateQueryBuilder = (model, requestBody, customWhere = null) => {
  if (!model) {
    throw new Error("Model is undefined or null.");
  }
  requestBody = JSON.parse(JSON.stringify(requestBody));
  const columns = Object.keys(model.rawAttributes);
  // Identify primary keys and auto increment keys
  const primaryKeyColumns = columns.filter(
    (column) =>
      model.rawAttributes[column].primaryKey ||
      model.rawAttributes[column].autoIncrement
  );
  // Filter columns to update, excluding primary keys, auto increment keys, and the 'created_by' column
  const filteredColumns = columns.filter((column) => {
    return (
      column !== "created_by" && // Exclude the 'created_by' column
      column !== "updated_at" && // Exclude the 'updated_at' column
      requestBody.hasOwnProperty(column) &&
      requestBody[column] !== undefined &&
      column !== "created_at" &&
      requestBody[column] != "null" &&
      // requestBody[column] !== "" &&
      !model.rawAttributes[column].primaryKey &&
      !model.rawAttributes[column].autoIncrement
    );
  });

  // Generate SET part of the update query
  const setStatements = filteredColumns
    .map((column) => `${column} = ?`)
    .join(", ");
  // Determine columns for the WHERE part of the update query
  const whereColumns =
    customWhere && Object.keys(customWhere).length > 0
      ? Object.keys(customWhere)
      : primaryKeyColumns;
  const whereClauses = whereColumns
    .filter((column) => requestBody.hasOwnProperty(column))
    .map((column) => `${column} = ?`);
  if (whereClauses.length === 0) {
    throw new Error("No columns provided for the WHERE clause.");
  }
  const whereClause = whereClauses.join(" AND ");
  const query = `UPDATE ${model.tableName} SET ${setStatements} WHERE ${whereClause}`;
  const values = filteredColumns.map((column) => {
    const attributeType = model.rawAttributes[column].type;
    if (
      (attributeType instanceof DataTypes.JSON ||
        attributeType instanceof DataTypes.TEXT) &&
      (typeof requestBody[column] === "object" ||
        Array.isArray(requestBody[column]))
    ) {
      return JSON.stringify(requestBody[column]);
    }
    if (requestBody[column] === "") {
      return null;
    }
    if (requestBody[column] == null) {
      return null;
    }
    return requestBody[column];
  });
  // Append WHERE clause values to the values array
  if (customWhere) {
    whereColumns.forEach((column) => {
      //   if (customWhere.hasOwnProperty(column)) {
      values.push(customWhere[column]);
      //   }
    });
  } else {
    primaryKeyColumns.forEach((column) => {
      if (requestBody.hasOwnProperty(column)) {
        values.push(requestBody[column]);
      }
    });
  }

  return {
    query: query,
    values: values,
  };
};
/**Function for validation helper  */
export const validateRequest = (schema, isParsed = true) => {
  return (req, res, next) => {
    try {
      if (!isParsed) {
        req.body = decodeAndParseFieldsSync([req.body])[0];
      }
      const { error } = schema.validate(req.body);
      if (error) {
        const errors = error.details.map((detail) => detail.message);
        return res.status(400).json({ status: false, message: errors[0] });
      }
      next();
    } catch (err) {
      console.error("Validation error:", err);
      return res
        .status(500)
        .json({ status: false, message: "Internal server error" });
    }
  };
};

/**Function to fetch plugin users record */
export const tableDataFetch = async (fieldName, value, tableName) => {
  try {
    const [userDataFetch] = await db.query(
      `SELECT * FROM ${tableName} where deleted = "0" AND ${fieldName} = '${value}'`
    );
    return userDataFetch[0];
  } catch (error) {
    return error;
  }
};
/**Function for error return message */
export const sendSuccessResponse = (res, statusCode, message) => {
  return res.status(statusCode).json({
    status: true,
    message: message,
  });
};
/**Function for error return message */
export const sendErrorResponse = (res, statusCode, message) => {
  return res.status(statusCode).json({
    status: false,
    message: message,
  });
};
/**Function to catch error */
export const catchErrorResponse = (res, error) => {
  /**Store error of record */
  storeError(error);
  return res.status(500).json({
    status: false,
    message: error.message,
  });
};
/**Function to upload file */
export const uploadFile = async (folder, file) => {
  const fileExtension = path.extname(file.name);
  const fileName = `${Date.now()}${fileExtension}`;
  const uploadPath = path.join(__dirname, `../public/${folder}/`, fileName);
  const folderPath = path.join(__dirname, `../public/${folder}/`);
  if (!fs.existsSync(folderPath)) {
    fs.mkdirSync(folderPath, { recursive: true });
  }
  const storePath = `${folder}/${fileName}`;
  await file.mv(uploadPath);
  console.log("storePath: ", storePath);
  return storePath;
};
/**Function to delete file from folder */
export const deleteFile = async (folderName) => {
  const oldImagePath = path.join(__dirname, `../public/${folderName}/`);
  if (fs.existsSync(oldImagePath)) {
    fs.rmSync(oldImagePath, { force: true });
  }
};
/**Function to make joins */
export const makeJoins = async (joins) => {
  let query = ``;
  if (joins && joins.length > 0) {
    joins.forEach((join) => {
      const { type, targetTable, onCondition } = join;
      query += ` ${type.toUpperCase()} JOIN ${targetTable} ON ${onCondition}`;
    });
  }
  return query;
};

/**Function to make name query */
export const makeNameQuery = async (record) => {
  const transformArrayToSQL = (arr) => {
    return arr
      .map((item) => {
        return item.fields
          .map((field) => {
            return `${item.table}.${field} as ${item.name}_${field}`;
          })
          .join(", ");
      })
      .join(", ");
  };

  const result = transformArrayToSQL(record);
  return result;
};
/**Where condition check*/

const skipArray = [
  // "organization",
  "legislation_doc",
  "meeting_notes",
  "objective_setting_target_plan",
  "policy_comments",
  "questions",
  "sidebar",
  "notification",
  "she_legal_register",
  "attendance_recording",
  "survey_design",
];

function sanitizeObject(obj) {
  for (const key in obj) {
    if (typeof obj[key] === "string") {
      // Replace ' and " with empty string
      obj[key] = obj[key].replace(/['"]/g, "");
    }
  }
  return obj;
}
// export const whereCondition = async (request) => {
//   const filterTable = request.filterTable ? request.filterTable : request.table;
//   const filterString = request.filter || "";
//   const columnName = request.columnName || "id";
//   const groupBy = request.groupBy;
//   const orderBy = request?.orderBy || "DESC";
//   try {
//     let filter = {};
//     if (typeof filterString === "string" && filterString.trim() !== "") {
//       filter = JSON.parse(filterString);
//     }
//     let idCheck = "Max(id)";
//     for (let key in filter) {
//       if (filter[key]) {
//         idCheck = "id";
//       }
//     }

//     const start =
//       request.grouped == "true" && !groupBy
//         ? `AND ${request.table}.id IN (SELECT ${idCheck} as id FROM ${request.table} WHERE deleted = 0 GROUP BY name) `
//         : "";
//     const page = parseInt(request.page) || 1;
//     const pageSize = parseInt(request.pageSize) || 10;
//     const offset = (page - 1) * pageSize;
//     const where =
//       start + (request.id && !isNaN(Number(request.id)) ? `AND ${request.table}.${columnName} = ${request.id}` : "");

//     let ownerCheck;
//     let user;
//     if (request.user) {
//       [user] = await db.query(`SELECT view_type FROM users WHERE id = ${request.user?.sessionid}`);
//       ownerCheck = request.ownerFilter ? `OR ${request.table}.${request.ownerFilter} = ${request.user.sessionid}` : "";
//     }

//     let accessType = "";
//     if (!request.user?.isSuperAdmin) {
//       if (request.user?.settings) {
//         const columnName =
//           request.table === "users" ? "my_organization" : request.table === "organization" ? "id" : "organization";

//         // Create JSON_CONTAINS conditions for each organization ID
//         const jsonContainsConditions = request.user?.additionalData
//           .map((orgId) => `JSON_CONTAINS(${request.table}.${columnName}, '${orgId}', '$')`)
//           .join(" OR ");

//         accessType = `AND (${jsonContainsConditions})`;
//       } else if (!skipArray.includes(request.table)) {
//         if (user && user[0] && user[0]?.view_type) {
//           if (user[0]?.view_type == "all") {
//             accessType = `AND ${request.table}.${
//               request.table === "users" ? "my_organization" : request.table === "organization" ? "id" : "organization"
//             } IN (${
//               request.user?.additionalData
//                 ? request.user?.additionalData
//                 : JSON.parse(request.user?.sessionOrganization)
//             })`;
//             if (request.table === "users" && request.id) {
//               accessType = "";
//             }
//           } else if (user[0]?.view_type === "team") {
//             const teamsQuery = `SELECT team_members FROM teams WHERE team_leader = ${request.user.sessionid}`;
//             // console.log(teamsQuery);
//             const teams = await db.query(teamsQuery);
//             const teamMembers = JSON.parse(teams[0]?.team_members || "[]");
//             teamMembers.push(request.user.sessionid);
//             if (teamMembers.length > 0) {
//               accessType = `AND (${request.table}.created_by IN (${teamMembers}) ${ownerCheck})`;
//             }
//             if (request.table === "users" && request.id) {
//               accessType = "";
//             }
//           } else if (user[0]?.view_type === "self" && request.user?.sessionid != request.id) {
//             accessType = `AND (${request.table}.created_by = ${request.user?.sessionid} ${ownerCheck})`;
//             if (request.table === "users" && request.id) {
//               accessType = "";
//             }
//           }
//         }
//       }
//     }

//     /** access type filter */
//     let whereFilter = "";
//     const whereFilterRecord = Object.entries(filter)
//       .filter(([key, value]) => value !== "" && value !== null)
//       .map(([key, value]) => {
//         if (key === "created_at") {
//           return `Date(${filterTable}.${key}) = '${value}'`;
//         } else {
//           return `${filterTable}.${key} = '${value}'`;
//         }
//       })
//       .join(" AND ");

//     if (whereFilterRecord !== "") {
//       whereFilter = `AND ${whereFilterRecord}`;
//     }

//     const pagination = request.all === "false" ? `LIMIT ${pageSize} OFFSET ${offset}` : "";
//     const returnQuery = `${accessType} ${where} ${whereFilter}  ORDER BY ${request.table}.id ${orderBy} ${pagination}`;
//     return returnQuery;
//   } catch (error) {
//     throw error;
//   }
// };

export const whereCondition = async (request) => {
  const filterTable = request.filterTable ? request.filterTable : request.table;
  const filterString = request.filter || "";
  const columnName = request.columnName || "id";
  const groupBy = request.groupBy;
  const orderBy = request?.orderBy || "DESC";
  try {
    let filter = {};
    if (typeof filterString === "string" && filterString.trim() !== "") {
      filter = JSON.parse(filterString);
    }
    let idCheck = "Max(id)";
    for (let key in filter) {
      if (filter[key]) {
        idCheck = "id";
      }
    }

    const start =
      request.grouped == "true" && !groupBy
        ? `AND ${request.table}.id IN (SELECT ${idCheck} as id FROM ${request.table} WHERE deleted = 0 GROUP BY name) `
        : "";
    const page = parseInt(request.page) || 1;
    const pageSize = parseInt(request.pageSize) || 10;
    const offset = (page - 1) * pageSize;
    const where =
      start +
      (request.id && !isNaN(Number(request.id))
        ? `AND ${request.table}.${columnName} = ${request.id}`
        : "");

    let ownerCheck;
    let user;
    if (request.user) {
      [user] = await db.query(
        `SELECT view_type FROM users WHERE id = ${request.user?.sessionid}`
      );
      ownerCheck = request.ownerFilter
        ? `OR ${request.table}.${request.ownerFilter} = ${request.user.sessionid}`
        : "";
    }

    let accessType = "";
    if (!request.user?.isSuperAdmin) {
      if (request.user?.settings) {
        const columnName =
          request.table === "users"
            ? "my_organization"
            : request.table === "organization"
              ? "id"
              : "organization";

        const jsonContainsConditions = request.user?.additionalData
          .map(
            (orgId) =>
              `JSON_CONTAINS(${request.table}.${columnName}, '${orgId}', '$')`
          )
          .join(" OR ");

        accessType = `AND (${jsonContainsConditions})`;
      } else if (!skipArray.includes(request.table)) {
        if (user && user[0] && user[0]?.view_type) {
          if (user[0]?.view_type == "all") {
            accessType = `AND ${request.table}.${request.table === "users"
              ? "my_organization"
              : request.table === "organization"
                ? "id"
                : "organization"
              } IN (${request.user?.additionalData
                ? request.user?.additionalData
                : JSON.parse(request.user?.sessionOrganization)
              })`;
            if (request.table === "users" && request.id) {
              accessType = "";
            }
          } else if (user[0]?.view_type === "team") {
            const teamsQuery = `SELECT team_members FROM teams WHERE team_leader = ${request.user.sessionid}`;
            const teams = await db.query(teamsQuery);
            const teamMembers = JSON.parse(teams[0]?.team_members || "[]");
            teamMembers.push(request.user.sessionid);
            if (teamMembers.length > 0) {
              accessType = `AND (${request.table}.created_by IN (${teamMembers}) ${ownerCheck})`;
            }
            if (request.table === "users" && request.id) {
              accessType = "";
            }
          } else if (
            user[0]?.view_type === "self" &&
            request.user?.sessionid != request.id
          ) {
            accessType = `AND (${request.table}.created_by = ${request.user?.sessionid} ${ownerCheck})`;
            if (request.table === "users" && request.id) {
              accessType = "";
            }
          }
        }
      }
    }

    /** Check if columns exist in the table before filtering */
    const [columnsResult] = await db.query(`SHOW COLUMNS FROM ${filterTable}`);
    const existingColumns = new Set(columnsResult.map((col) => col.Field));

    let whereFilter = "";
    const invalidFilters = [];
    const whereFilterRecord = Object.entries(filter)
      .filter(([key, value]) => {
        if (value !== "" && value !== null) {
          if (!existingColumns.has(key)) {
            invalidFilters.push(key);
            return false;
          }
          return true;
        }
        return false;
      })
      .map(([key, value]) => {
        if (key === "created_at") {
          return `Date(${filterTable}.${key}) = '${value}'`;
        } else {
          return `${filterTable}.${key} = '${value}'`;
        }
      })
      .join(" AND ");

    if (invalidFilters.length > 0) {
      console.log("Invalid filter columns:", invalidFilters);
    }

    if (whereFilterRecord !== "") {
      whereFilter = `AND ${whereFilterRecord}`;
    }

    const pagination =
      request.all === "false" ? `LIMIT ${pageSize} OFFSET ${offset}` : "";
    const returnQuery = `${accessType} ${where} ${whereFilter} ORDER BY ${request.table}.id ${orderBy} ${pagination}`;
    return returnQuery;
  } catch (error) {
    throw error;
  }
};

/**Function to make name query */
export const searchConditionRecord = async (search, searchTableName) => {
  search = search?.trim();
  if (!search) return "";

  search = search.trim();
  const conditions = searchTableName
    .map((field) => `${field} LIKE '%${search}%'`)
    .join(" OR ");
  return `AND (${conditions}) `;
};

/**Function to delete record according to table*/
export const deleteRecord = async (table, id, columnName = "id", sessionId) => {
  console.log("table: ", table);
  try {
    if (typeof table !== "string") {
      table = table.tableName;
    }

    if (table === "organization") {
      const tableRecordArray = {
        organization: "parent_id",
        users: "my_organization",
      };

      let dataCheck = false;

      const [checkUser] = await db.query(
        `SELECT is_super_admin , my_organization FROM users WHERE id = ?`,
        [sessionId]
      );

      if (checkUser.length === 0) {
        return { status: false, message: "User not found" };
      }

      if (!checkUser[0].is_super_admin) {
        const [checkRecord] = await db.query(
          `SELECT created_by FROM ${table} WHERE ${columnName} = ?`,
          [id]
        );

        if (checkRecord.length === 0) {
          return { status: false, message: "Record not found" };
        }

        if (checkRecord[0].created_by !== sessionId) {
          return {
            status: false,
            message: "You are not authorized to delete this record",
          };
        }
      }

      if (id == checkUser[0].my_organization) {
        return {
          status: false,
          message: "You cannot delete your own organization",
        };
      }

      for (const [tableName, fieldName] of Object.entries(tableRecordArray)) {
        const [checkRecordExist] = await db.query(
          `SELECT 1 FROM ${tableName} WHERE ${fieldName} = ? AND deleted = '0' LIMIT 1`,
          [id]
        );

        if (checkRecordExist.length > 0) {
          dataCheck = true;
          break;
        }
      }

      if (dataCheck) {
        return { status: false, message: "Record is in use, can't delete" };
      } else {
        await db.query(
          `UPDATE ${table} SET deleted = '1' WHERE ${columnName} = ${id}`
        );
        return { status: true, message: "Record deleted successfully" };
      }
    } else {
      const deleteQuery = `UPDATE ${table} SET deleted = '1' WHERE ${columnName} = ?`;
      const [result] = await db.query(deleteQuery, [id]);

      return { status: true, message: "Record deleted successfully" };
    }
  } catch (error) {
    storeError(error);
    throw error;
  }
};

/**Function to encode data */
export const encodeData = async (data) => {
  if (data) {
    return he.encode(data);
  }
  return null;
};
/**Function to decode data */
export const decodeData = async (data) => {
  if (data) {
    return he.decode(data);
  }
  return null;
};
/**Function to make decode of that record which is get from table */
export const FetchedResultDecodeData = async (
  data,
  returnName,
  columnName = "description"
) => {
  try {
    for (const record of data) {
      const recordData = he.decode(record[columnName]);
      record[returnName] = recordData;
    }
  } catch (error) {
    storeError(error);
    return;
  }
};
/**Function to check that coming record is already exist in same organization or not */
export const checkOrganizationRecordExist = async (request) => {
  const [recordExist] = await db.query(
    `SELECT * FROM ${request.table} WHERE ${request.fieldName} = '${request.value}' AND ${request.organizationFieldName} = '1' AND deleted = '0'`
  );
  return recordExist[0];
};
/**Function to Get parent id level */
export const getParentLevel = async (parentId) => {
  const [parentLevel] = await db.query(
    `SELECT level FROM organization WHERE id = ${parentId}`
  );
  return parentLevel.length ? parentLevel[0].level : 0;
};

export const calculateAgeFromLicense = async (license) => {
  // console.log(license);
  const dobString = license?.substring(0, 6);

  const year = parseInt(dobString.substring(0, 2));
  const month = parseInt(dobString.substring(2, 4)) - 1;
  const day = parseInt(dobString.substring(4, 6));

  const currentYear = new Date().getFullYear();
  const currentTwoDigitYear = currentYear % 100;
  const century = year <= currentTwoDigitYear ? 2000 : 1900;
  const fullYear = century + year;

  const dob = new Date(fullYear, month, day);
  // console.log(dob);

  const today = new Date();

  let age = today.getFullYear() - dob.getFullYear();
  const monthDifference = today.getMonth() - dob.getMonth();
  const dayDifference = today.getDate() - dob.getDate();

  if (monthDifference < 0 || (monthDifference === 0 && dayDifference < 0)) {
    age--;
  }

  return age;
};

/**Count Query Condition */
export const countQueryCondition = async (Query) => {
  try {
    const totalCountQuery = Query.substring(0, Query.indexOf("ORDER BY"));
    const [query] = await db.query(totalCountQuery);
    return query.length;
  } catch (error) {
    storeError(error);
    return;
  }
};

/**Function to edit Audit Template Questions */
export const insertUpdateQuestions = async (
  audit_template_id,
  sessionId,
  questions,
  type
) => {
  try {
    if (type === "delete") {
      const deleteQuestionsQuery = `UPDATE questions SET deleted = 1 WHERE audit_template_id = ?`;
      const deleteQuestionsValues = [audit_template_id];
      const [deleteQuestions] = await db.query(
        deleteQuestionsQuery,
        deleteQuestionsValues
      );
    } else {
      for (let question of questions) {
        if (type === "update") {
          /** check that question no is decreased or not */
          const questionNo = questions.length;
          const deleteQuestionQuery = `UPDATE questions SET deleted = 1 WHERE audit_template_id = ? AND question_no > ? AND deleted = 0`;
          const deleteQuestionValues = [audit_template_id, questionNo];
          const [deleteQuestion] = await db.query(
            deleteQuestionQuery,
            deleteQuestionValues
          );
          const updateQuestionsQuery = `UPDATE questions SET question_title = ?, answer_type = ?, answer_key = ?, points = ?, options = ?, min = ?, max = ?, required = ?, updated_by = ? WHERE audit_template_id = ? AND question_no = ? AND deleted = 0`;
          const updateQuestionsValues = [
            question.question_title,
            question.answer_type,
            JSON.stringify(question.answer_key),
            question.points ? JSON.stringify(question.points) : "0",
            JSON.stringify(question.options),
            question.min,
            question.max,
            question.required ?? null,
            sessionId,
            audit_template_id,
            question.question_no,
          ];
          const [updateQuestions] = await db.query(
            updateQuestionsQuery,
            updateQuestionsValues
          );
          if (updateQuestions.affectedRows === 0) {
            /**Insert record audit template questions */
            const createQuestionQuery = `INSERT INTO questions (question_title, audit_template_id, question_no, answer_type, answer_key, points, options, min, max, required, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
            const createQuestionValues = [
              question.question_title,
              audit_template_id,
              question.question_no,
              question.answer_type,
              JSON.stringify(question.answer_key),
              question.points ? JSON.stringify(question.points) : "0",
              JSON.stringify(question.options),
              question.min,
              question.max,
              question.required ?? null,
              sessionId,
            ];
            const [createQuestion] = await db.query(
              createQuestionQuery,
              createQuestionValues
            );
          }
        } else {
          /** Default Create Questions */
          /**Insert record for audit template*/
          const createQuestionsQuery = `INSERT INTO questions (question_title, audit_template_id, question_no, answer_type, answer_key, points, options, min, max, required, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
          const createQuestionsValues = [
            question.question_title,
            audit_template_id,
            question.question_no,
            question.answer_type,
            JSON.stringify(question.answer_key),
            question.points ? JSON.stringify(question.points) : "0",
            JSON.stringify(question.options),
            question.min,
            question.max,
            question.required ?? null,
            sessionId,
            question.id,
          ];
          const [createQuestions] = await db.query(
            createQuestionsQuery,
            createQuestionsValues
          );
        }
      }
    }
  } catch (error) {
    storeError(error);
    return;
  }
};

/**Function to Get Audit Template Questions list */
export const getQuestions = async (audit_template_id) => {
  try {
    const questionsQuery = `SELECT * FROM questions WHERE audit_template_id = ? AND deleted = 0`;
    const [questions] = await db.query(questionsQuery, [audit_template_id]);
    /** Parse the options field of each question */
    questions.forEach((question) => {
      question.options = JSON.parse(question?.options || "[]");
      question.answer_key = JSON.parse(question?.answer_key || "[]");
      question.points = JSON.parse(question.points);
    });
    return questions;
  } catch (error) {
    storeError(error);
    return;
  }
};

/**Function to get the name of user */
export const getUserName = async (id) => {
  const [result] = await db.query(`SELECT name FROM users WHERE id = ?`, [id]);
  return result[0]?.name ?? "";
};

/**Function to get tables record details according to given record */
export const makeLoopAndGetMultipleUsersDetails = async (
  meetings,
  returnName
) => {
  /**Make loop for getting record of teams members details according to team leaders */
  for (const team of meetings) {
    const teamMembers = JSON.parse(team.participants);
    if (teamMembers.length > 0) {
      const teamsMembersQuery = `SELECT id,name,surname,profile,organization FROM users WHERE id IN (${teamMembers.join(
        ", "
      )})`;
      const [teamsMembers] = await db.query(teamsMembersQuery);
      const team_leader = await getUserName(team.team_leader);
      const created_by = await getUserName(team.created_by);
      team.team_leader_name = team_leader;
      team.created_by_name = created_by;
      team.team_members = teamsMembers;
      team[returnName] = teamsMembers;
    }
  }
  return meetings;
};

export const getStringOrTextFields = (model) => {
  // Get the attributes from the model
  const attributes = model.rawAttributes;

  // Get the table name from the model
  const tableName = model.tableName;

  // Filter fields that are of type STRING or TEXT and format as tableName.field
  const stringOrTextFields = Object.keys(attributes)
    .filter((key) => {
      const type = attributes[key].type;
      return type instanceof DataTypes.STRING || type instanceof DataTypes.TEXT;
    })
    .map((field) => `${tableName}.${field}`);

  return stringOrTextFields;
};

export const dateValidator = (date) => {
  try {
    return !isNaN(new Date(date).getTime()) ? date : null;
  } catch (error) {
    storeError(error);
    return null;
  }
};

export const tableRecord = async (field, value, table) => {
  try {
    const tableDataRecordQuery = `SELECT * FROM ${table} WHERE deleted = "0" AND ${field} = ?`;
    const [tableDataRecord] = await db.query(tableDataRecordQuery, [value]);
    return tableDataRecord;
  } catch (error) {
    storeError(error);
    return;
  }
};

/**function to encode html content */
export const encodeTheEditorContent = async (data, key = "text") => {
  if (data[0]?.[key]) {
    return data.map((obj) => ({ ...obj, [key]: he.encode(obj[key]) }));
  }
  return [{}];
};

export const checkPolicyUpdated = async (policy_id, data, req) => {
  const {
    organization,
    department,
    id,
    policy_title,
    policy_description,
    policy_purpose,
    policy_scope,
    policy_definitions,
    policy_establishment_date,
    policy_owner,
    policy_version_no,
    policy_next_review_date,
    role_responsibilities,
    policy_statement,
    compliance_enforcement,
    reference_appendices,
    status,
    send_to_employee,
  } = data;

  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
  }

  const encodedPolicy_statement = policy_statement
    ? await encodePolicy_statement(policy_statement)
    : "";

  const encodedReference_appendices = reference_appendices
    ? await encodeReference_appendices(reference_appendices)
    : "";

  const policyIdQuery = `SELECT * , DATE_FORMAT(policy_establishment_date, '%Y-%m-%d') as policy_establishment_date, DATE_FORMAT(policy_next_review_date, '%Y-%m-%d') as policy_next_review_date FROM policy WHERE id = ?`;
  const [policyRecord] = await db.query(policyIdQuery, [policy_id]);
  let isUpdated = false;
  if (policyRecord[0]?.policy_title != policy_title) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_description != policy_description) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_purpose != policy_purpose) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_scope != policy_scope) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_definitions != policy_definitions) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_establishment_date != policy_establishment_date) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_owner != policy_owner) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_next_review_date != policy_next_review_date) {
    isUpdated = true;
  }

  if (
    policyRecord[0]?.role_responsibilities !=
    JSON.stringify(role_responsibilities)
  ) {
    isUpdated = true;
  }

  if (policyRecord[0]?.policy_statement != encodedPolicy_statement) {
    isUpdated = true;
  }

  if (
    policyRecord[0]?.compliance_enforcement !=
    JSON.stringify(compliance_enforcement)
  ) {
    isUpdated = true;
  }

  if (
    policyRecord[0]?.reference_appendices !=
    JSON.stringify(encodedReference_appendices)
  ) {
    isUpdated = true;
  }

  if (policyRecord[0]?.organization != organizationId) {
    isUpdated = true;
  }

  if (policyRecord[0]?.department != department) {
    isUpdated = true;
  }

  /** If isUpdated true is updated then insert new record with new version */
  const policyVersionNo = generateNextVersion(
    policyRecord[0]?.policy_version_no || ""
  );
  if (req.body?.confirmation) {
    // const policyId = await uniqueIdGenerator(
    //   organizationId,
    //   department,
    //   "POL",
    //   "policy",
    //   "policy_id"
    // );
    req.body.created_by = req.user.sessionid;
    const policyId = policyRecord[0]?.policy_id;
    req.body.policy_id = policyId;
    req.body.reference_appendices = encodedReference_appendices;
    req.body.policy_statement = encodedPolicy_statement;
    req.body.policy_version_no = policyVersionNo;
    req.body.organization = organizationId;
    const { query, values } = createQueryBuilder(Policy, req.body);
    await db.query(query, values);
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Policy ",
      `This user created a new policy with title ${policy_title} for organization ${organization}`
    );
    /**Insert record for activity log */
  }
  return req.body?.confirmation;
};

/**function to decode the stored html content of text editor */
export const decodeTheEditorContent = async (data, key = "text") => {
  if (JSON.parse(data).length > 0 && JSON.parse(data)[0]?.[key]) {
    // console.log('come')
    return JSON.parse(data).map((obj) => ({
      ...obj,
      [key]: he.decode(obj[key]),
    }));
  }
  return [{}];
};

/**Function to get tables record details according to given record */
export const insertMeetingActionNotes = async (
  meeting_id,
  sessionId,
  notesRecord,
  meetingRecordingId,
  model = MeetingNotes
) => {
  for (let notes of notesRecord) {
    const encodedMeetingDiscussion = await encodeSingle_statement(
      notes.meeting_discussion
    );
    notes.meeting_discussion = encodedMeetingDiscussion;
    const encodedMeetingActionDescription = await encodeSingle_statement(
      notes.meeting_action_description
    );
    notes.meeting_action_description = encodedMeetingActionDescription;
    notes.meeting_id = meeting_id;
    notes.meeting_recording_id = meetingRecordingId;
    if (notes.id) {
      notes.updated_by = sessionId;
      /** Update the meetings notes */
      const { query, values } = updateQueryBuilder(model, notes);
      await db.query(query, values);
    } else {
      notes.created_by = sessionId;
      /**Insert record for Meeting Recording */
      const { query, values } = createQueryBuilder(model, notes);
      await db.query(query, values);
    }
  }
};

export const isTeamLeader = async (teamLeaderId, tableName, type) => {
  const getUserDetailsQuery = `SELECT id , role, name FROM users WHERE id = ${teamLeaderId}`;
  const getUserDetails = await db.query(getUserDetailsQuery);
  const query = `select id,name,team_members from teams where deleted='0' and team_leader = ${teamLeaderId}`;
  const result = await db.query(query);
  let returnQuery = "";
  if (result.length > 0 && getUserDetails[0]?.role != 0) {
    const team_members = JSON.parse(result[0]?.team_members || "[]");
    team_members.push(teamLeaderId);
    returnQuery = `${tableName}.created_by IN (${team_members})`;
  } else if (getUserDetails[0]?.role != 0) {
    returnQuery = `${tableName}.created_by = ${teamLeaderId}`;
  }
  if (type === "meeting" && getUserDetails[0]?.role != 0) {
    returnQuery = `(${returnQuery.length > 0 ? returnQuery + ` OR ` : ""
      } JSON_CONTAINS(participants, ${teamLeaderId}))`;
  } else if (type === "meeting_recording" && getUserDetails[0]?.role != 0) {
    /**meeting_recording */
    returnQuery = `(${returnQuery.length > 0 ? returnQuery + ` OR ` : ""}
    JSON_CONTAINS(${tableName}.participants, '{"id": ${teamLeaderId}}', '$'))`;
  }
  return returnQuery.length > 0 ? `AND ` + returnQuery : "";
};

/** Extract text from html content */
export const textExtractor = (html) => {
  // Load HTML content into cheerio
  const $ = load(html);
  // Extract text content
  const text = $("body").text().trim(); // Get text content excluding any script/style tags
  return text;
};

/**Function to get tables record details according to given record */
export const addUpdateObjectiveManagementPlan = async (
  objectiveId,
  sessionId,
  objectiveTargetRecord
) => {
  try {
    for (let objectiveTarget of objectiveTargetRecord) {
      if (objectiveTarget.id) {
        const updateObjectiveSettingTargetPlanQuery = `UPDATE objective_setting_target_plan SET action_title = ?, action_description = ?, responsible_department = ?, responsible_person = ?, start_date = ?, due_date = ?, action_progress_description = ?, action_progress_status = ? , updated_by = ? , objective_setting_id = ?  , organization = ? WHERE id = ?;`;
        const updateObjectiveSettingTargetPlanValues = [
          objectiveTarget.action_title,
          objectiveTarget.action_description,
          objectiveTarget.responsible_department,
          objectiveTarget.responsible_person,
          objectiveTarget.start_date,
          objectiveTarget.due_date,
          objectiveTarget.action_progress_description,
          objectiveTarget.action_progress_status,
          sessionId,
          objectiveId,
          objectiveTarget.organization,
          objectiveTarget.id,
        ];
        const updateObjectiveSettingTargetPlan = await db.query(
          updateObjectiveSettingTargetPlanQuery,
          updateObjectiveSettingTargetPlanValues
        );
      } else {
        /**Insert record for management plans */

        /** Generate Unique action plan id using uniqueIdGenerator */

        const insertObjectiveSettingTargetPlanQuery = `INSERT INTO objective_setting_target_plan (objective_setting_id , action_title, action_description, responsible_department, responsible_person, start_date, due_date, action_progress_description, action_progress_status ,created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ? ,? ,?);`;
        const insertObjectiveSettingTargetPlanValues = [
          objectiveId,
          objectiveTarget.action_title,
          objectiveTarget.action_description,
          objectiveTarget.responsible_department,
          objectiveTarget.responsible_person,
          objectiveTarget.start_date,
          objectiveTarget.due_date,
          objectiveTarget.action_progress_description,
          objectiveTarget.action_progress_status,
          sessionId,
        ];
        const insertObjectiveSettingTargetPlan = await db.query(
          insertObjectiveSettingTargetPlanQuery,
          insertObjectiveSettingTargetPlanValues
        );
      }
    }
  } catch (error) {
    storeError(error);
    return;
  }
};

export const encodeReference_appendices = async (data) => {
  try {
    if (data[0]?.reference) {
      return data.map((obj) => ({ reference: he.encode(obj.reference) }));
    }
    return [{}];
  } catch (error) {
    storeError(error);
    return;
  }
};

/**function to decode Reference_appendices html content */
export const decodeReference_appendices = async (data) => {
  try {
    if (data[0]?.reference) {
      return data.map((obj) => ({ reference: he.decode(obj.reference) }));
    }
    return [{}];
  } catch (error) {
    storeError(error);
    return;
  }
};

export const encodePolicy_statement = async (data) => {
  try {
    if (data) {
      return he.encode(data);
    }
    return null;
  } catch (error) {
    storeError(error);
    return;
  }
};
export const decodePolicy_statement = async (data) => {
  try {
    if (data) {
      return he.decode(data);
    }
    return null;
  } catch (error) {
    storeError(error);
    return;
  }
};

export const insertNotification = async (
  subject,
  message,
  user,
  messageType,
  sender
) => {
  try {
    const insertNotificationQuery = `INSERT INTO notification(subject, message, user_id, type, created_by) VALUES (?, ?, ?, ?, ?)`;
    await db.query(insertNotificationQuery, [
      subject,
      message,
      user,
      messageType,
      sender,
    ]);
    return true;
  } catch (error) {
    storeError(error);
    return false;
  }
};

export const policyIdGenerator = async () => {
  try {
    const policyIdQuery = `SELECT id, policy_id FROM policy ORDER BY id DESC LIMIT 1`;
    const [policyRecord] = await db.query(policyIdQuery);
    const policyId = policyRecord[0]?.policy_id;
    return `P${+policyId.split("P")[1] + 1}`;
  } catch (error) {
    storeError(error);
    return;
  }
};

export function generateNextVersion(previousVersion) {
  const pattern = /^\d+\.\d+\.\d+$/; // Pattern to match x.y.z format
  const isValidFormat = pattern.test(previousVersion);

  if (!isValidFormat) {
    // If previous version doesn't follow the pattern, start from 0.0.1
    return "0.0.1";
  }

  const versionParts = previousVersion.split(".").map(Number);
  const lastIndex = versionParts.length - 1;

  versionParts[lastIndex]++;

  for (let i = lastIndex; i > 0; i--) {
    if (versionParts[i] === 10) {
      versionParts[i] = 0;
      versionParts[i - 1]++;
    } else {
      break;
    }
  }

  const nextVersion = versionParts.join(".");

  return nextVersion;
}

export const getLevelDetails = async (id) => {
  try {
    const [result] = await db.query(
      `SELECT id,name,profile FROM users WHERE id = ?`,
      [id]
    );
    return result[0] ?? [];
  } catch (error) {
    storeError(error);
    return;
  }
};

// export const pointsCalculate = async (questions) => {
//   let totalPoints = 0;
//   let pointsEarned = 0;
//   for (let question of questions) {
//     if(Number(question.points)){
//       totalPoints += question.points;
//     }
//     else if (Array.isArray(question.points)) {
//       totalPoints += question.points.reduce((a, b) => Number(a) + Number(b), 0);
//     }
//     totalPoints += question.points;
//     if (
//       question.answer_type === "Multiple Choice" ||
//       question.answer_type === "Dropdown" ||
//       question.answer_type === "Checkbox"
//     ) {
//       const options = question?.options;
//       for (let option of options) {
//         let answer_key = question.answer_key;
//         /** answer_key is an array of index of options which are selected as a answer */
//         if (answer_key.includes(question.value)) {
//           pointsEarned += question.points;
//           break;
//         }
//       }
//     } else if (question.answer_type === "Paragraph") {
//       if (question.value) {
//         pointsEarned += question.points;
//       }
//     } else if (
//       question.answer_type === "Date" ||
//       question.answer_type === "Time"
//     ) {
//       if (question.value && question.value === question.answer_key) {
//         pointsEarned += question.points;
//       }
//     } else if (question.answer_type === "Linear Scale") {
//       if (
//         question.value &&
//         question.value >= question.min &&
//         question.value <= question.max
//       ) {
//         pointsEarned += question.points;
//       }
//     }
//   }
//   return { totalPoints, pointsEarned };
// };

// export const pointsCalculate = async (sections) => {
//   let totalPoints = 0;
//   let pointsEarned = 0;

//   for (let section of sections) {
//     for (let question of section.questions) {
//       // Calculate total points
//       if (Number(question.points)) {
//         totalPoints += question.points;
//       } else if (Array.isArray(question.points)) {
//         totalPoints += question.points.reduce(
//           (a, b) => Number(a) + Number(b),
//           0
//         );
//       }

//       // Points earned calculation based on answer_type
//       if (
//         question.answer_type === "Multiple Choice" ||
//         question.answer_type === "Dropdown" ||
//         question.answer_type === "Checkbox"
//       ) {
//         const options = question?.options;
//         const answer_key = question.answer_key;
//         /** answer_key is assumed to be an array of selected answer indexes */
//         if (Array.isArray(answer_key)) {
//           for (let i = 0; i < options.length; i++) {
//             if (answer_key.includes(i) && options[i] === question.value) {
//               if (Number(question.points)) {
//                 pointsEarned += Number(question.points);
//               } else if (Array.isArray(question.points)) {
//                 pointsEarned += question.points.reduce(
//                   (a, b) => Number(a) + Number(b),
//                   0
//                 );
//               }
//               break;
//             }
//           }
//         }
//       } else if (question.answer_type === "Paragraph") {
//         if (question.value) {
//           pointsEarned += Number(question.points);
//         }
//       } else if (
//         question.answer_type === "Date" ||
//         question.answer_type === "Time"
//       ) {
//         if (question.value && question.value === question.answer_key) {
//           pointsEarned += Number(question.points);
//         }
//       } else if (question.answer_type === "Linear Scale") {
//         if (
//           question.value &&
//           question.value >= question.min &&
//           question.value <= question.max
//         ) {
//           pointsEarned += Number(question.points);
//         }
//       }
//     }
//   }
//   return { totalPoints, pointsEarned };
// };

export const pointsCalculate = async (sections) => {
  let totalPoints = 0;
  let pointsEarned = 0;

  for (let section of sections) {
    for (let question of section.questions) {
      // Calculate total points
      if (Number(question.points)) {
        totalPoints += question.points;
      } else if (Array.isArray(question.points)) {
        const sectionPoints = question.points.reduce(
          (a, b) => Number(a) + Number(b),
          0
        );
        totalPoints += sectionPoints;
      }

      // Points earned calculation based on answer_type
      if (
        question.answer_type === "Multiple Choice" ||
        question.answer_type === "Dropdown" ||
        question.answer_type === "Checkbox"
      ) {
        const options = question?.options;
        const answer_key = question.answer_key;
        const value = question.value;

        if (Array.isArray(answer_key)) {
          for (let i = 0; i < answer_key.length; i++) {
            const isValueMatch = (
              (!Array.isArray(value) && value === answer_key[i]) ||

              (Array.isArray(value) &&
                (value.includes(answer_key[i]) ||
                  value.includes(options[answer_key[i]]) ||
                  value.indexOf(options[answer_key[i]]) !== -1))
            );

            if (isValueMatch) {
              const pointsToAdd = Array.isArray(question.points)
                ? question.points[answer_key[i]]
                : question.points;

              pointsEarned += Number(pointsToAdd);
            }
          }
        }
      } else if (question.answer_type === "Paragraph") {
        if (question.value) {
          pointsEarned += Number(question.points);
        }
      } else if (
        question.answer_type === "Date" ||
        question.answer_type === "Time"
      ) {
        if (question.value && question.value === question.answer_key) {
          pointsEarned += Number(question.points);
        }
      } else if (question.answer_type === "Linear Scale") {
        if (
          question.value &&
          question.value >= question.min &&
          question.value <= question.max
        ) {
          pointsEarned += Number(question.points);
        }
      }
    }
  }

  return { totalPoints, pointsEarned };
};



export const addLegislationDoc = async (
  legislationId,
  organizationId,
  department,
  request
) => {
  try {
    const { record, document_file } = request.body;
    // const { document } = request.files;
    for (const documentRecord of document_file) {
      // const file = document[documentRecord];
      // const index = getIndexFromKey(documentRecord);
      // const title = record[`document_file[${index}][title]`];
      // const updatePath = await uploadFile("legislation_docs", file);
      // const ddrm_id = await uploadToDDRM(req.body.sidebar_id, file, req);
      const ddrm_id = documentRecord.ddrm_id;
      /**Insert record for Legislation */
      const createLegislationQuery = `INSERT INTO legislation_doc (legislation_id, ddrm_id ,old_version , new_version , organization, department) VALUES (?, ?, ?, ? , ? , ?);`;
      const createLegislationValues = [
        legislationId,
        ddrm_id,
        "0.000",
        "0.001",
        organizationId,
        department,
      ];
      await db.query(createLegislationQuery, createLegislationValues);
    }
  } catch (error) {
    storeError(error);
    return;
  }
};

/**Function to generate index key data according to given multiple docs  */
const getIndexFromKey = (key) => {
  const match = key.match(/\[(\d+)\]/);
  return match ? parseInt(match[1]) : -1; // Return -1 if index not found
};

const depthDelete = async (req, res) => {
  try {
    const { id = 8 } = req.params;
    const { table = "users" } = req.body;
    //  let modelObj= {
    //   "users": User,
    //   "teams": Team,
    //   "roles": Role,
    //   "meetings":Meeting,
    //   "meeting_executions":MeetingRecording,
    //  }
    let modelObj = {};
    const schemaDir = "sequelize";
    const files = fs.readdirSync(schemaDir);
    // console.log(files.length);
    for (let i = 0; i < files.length; i++) {
      const file = files[i];
      // console.log(file);
      if (file.endsWith("Schema.js")) {
        const filePath = path
          .join(process.cwd(), schemaDir, file)
          .replace(/\\/g, "/");
        const module = await import(`file://${filePath}`);
        const model = module.default;
        // console.log(model.tableName, "model");
        modelObj[model.tableName] = model;
      } else {
        continue;
      }
    }

    for (let key in modelObj) {
      const modelStructure = modelObj[key].getAttributes();
      const columns = Object.keys(modelStructure);
      for (let column of columns) {
        if (
          modelStructure[column].references &&
          modelStructure[column].references.tableName === table
        ) {
          const updateQuery = `update ${modelObj[key].tableName} set deleted= 1 where ${column} = ${id}`;
          const [updatedRecord] = await db.query(updateQuery);
        }
      }
    }
  } catch (error) {
    console.log(error);
  }
};

const buildGraph = (tasks) => {
  const graph = new Map();
  const inDegree = new Map();

  tasks.forEach((task) => {
    const {
      id: taskName,
      successor_task_template,
      predecessor_task_template,
    } = task;

    if (
      successor_task_template === taskName ||
      predecessor_task_template === taskName
    ) {
      return;
    }

    if (!graph.has(taskName)) graph.set(taskName, []);
    if (!inDegree.has(taskName)) inDegree.set(taskName, 0);

    // Process successor
    if (successor_task_template) {
      if (!graph.has(successor_task_template))
        graph.set(successor_task_template, []);
      graph.get(taskName).push(successor_task_template);

      if (!inDegree.has(successor_task_template))
        inDegree.set(successor_task_template, 0);

      inDegree.set(
        successor_task_template,
        (inDegree.get(successor_task_template) || 0) + 1
      );
    }

    // Process predecessor
    if (predecessor_task_template) {
      if (!graph.has(predecessor_task_template))
        graph.set(predecessor_task_template, []);
      // graph.get(taskName).push(predecessor_task_template);
      if (!inDegree.has(predecessor_task_template))
        inDegree.set(predecessor_task_template, 0);
    }
  });

  // console.log("Graph:", graph);
  // console.log("In-Degree:", inDegree);

  return { graph, inDegree };
};

const topologicalSortWithLevels = (graph, inDegree) => {
  const levels = [];
  const queue = [];

  // Add all nodes with in-degree 0 to the queue
  for (const [node, degree] of inDegree.entries()) {
    if (degree === 0) queue.push(node);
  }

  while (queue.length > 0) {
    const level = [];
    const nextQueue = [];

    // Process all nodes in the current level
    while (queue.length > 0) {
      const node = queue.shift();
      level.push(node);

      for (const neighbor of graph.get(node)) {
        inDegree.set(neighbor, inDegree.get(neighbor) - 1);
        if (inDegree.get(neighbor) === 0) nextQueue.push(neighbor);
      }
    }

    levels.push(level);
    queue.push(...nextQueue);
  }

  // Check for cycles
  const totalNodes = [...graph.keys()].length;
  const processedNodes = levels.reduce((acc, level) => acc + level.length, 0);
  if (processedNodes !== totalNodes) {
    console.warn("Cycle detected in tasks, topological sort not possible.");
  }

  return levels;
};

export const checkCircularDependency = async (
  currentId,
  predecessorId,
  successorId,
  tableName = "action_template_list"
) => {
  const visited = new Set();

  async function dfs(taskId, isForward) {
    if (visited.has(taskId)) {
      return true; // Cycle detected
    }

    visited.add(taskId);

    const [task] = await db.query(
      `SELECT predecessor_task_template, successor_task_template FROM ${tableName} WHERE id = ?`,
      [taskId]
    );

    if (task.length > 0) {
      const { predecessor_task_template, successor_task_template } = task[0];

      if (isForward) {
        if (successor_task_template) {
          if (
            successor_task_template === predecessorId ||
            (await dfs(successor_task_template, isForward))
          ) {
            return true;
          }
        }
      } else {
        if (predecessor_task_template) {
          if (
            predecessor_task_template === successorId ||
            (await dfs(predecessor_task_template, isForward))
          ) {
            return true;
          }
        }
      }
    }

    visited.delete(taskId);
    return false;
  }

  // Check forward chain starting from predecessorId
  if (predecessorId && (await dfs(predecessorId, true))) {
    return true;
  }

  // Reset visited set
  visited.clear();

  // Check backward chain starting from successorId
  if (successorId && (await dfs(successorId, false))) {
    return true;
  }

  return false;
};

export const orderTasks = async (tasks) => {
  try {
    const { graph, inDegree } = buildGraph(tasks);
    const levels = topologicalSortWithLevels(graph, inDegree);
    return levels;
  } catch (error) {
    console.warn("Error ordering tasks:", error);
  }
};

export const sortTask = async (tasks) => {
  // Find that tasks whose successor and predecessor is null
  for (let task of tasks) {
    if (!task.successor_task_template && !task.predecessor_task_template) {
    }
  }
};

export const isChild = async (currentid, targetParentId, model) => {
  const children = await getRecord(model, "parent_id", currentid);
  for (const child of children) {
    if (child.id.toString() === targetParentId.toString()) {
      return true;
    }
    // Recursively check if any of the children have the targetParentId as a child
    const childCheck = await isChild(child.id, targetParentId);
    if (childCheck) {
      return true;
    }
  }
  return false;
};
export async function validateParentId(id, parentId, model) {
  // Fetch the current record and its children
  const record = await getRecord(model, "id", id);
  if (record.length === 0) {
    throw new Error("Record not found.");
  }

  // Recursive function to check if parentId is a child of the current record

  // Validate if parentId is a child
  const isInvalidParent = await isChild(id, parentId, model);
  if (isInvalidParent) {
    throw new Error("The parent  cannot be one of its children.");
  }
  return true; // Parent ID is valid
}

import chalk from "chalk";
import AuditLog from "../sequelize/AuditLogSchema.js";
import { uploadToDDRM } from "./ddrmUploader.js";
import moment from "moment";
import FireEquipmentStatusHistory from "../sequelize/fireEquipmentStatusHistorySchema.js";
import CustomActionCreation from "../sequelize/CustomActionCreationSchema.js";
import ContractorRegistration from "../sequelize/ContractorRegistrationSchema.js";
import sequelize from "../sequelize/sequelize.js";
import axios from "axios";
import sendEmail from "./sendEmail.js";
import { auditLinkedTable } from "../constants/constants.js";

export const whereConditionHelper = ({
  id,
  filter,
  grouped,
  search,
  model,
  includes = [],
  table,
}) => {
  const where = { deleted: 0 };

  // ID filtering
  if (id) where[`${table}.id`] = id;

  // Parse filter string
  if (filter) {
    const parsedFilter = JSON.parse(filter);
    Object.entries(parsedFilter).forEach(([key, value]) => {
      where[key] = value;
    });
  }

  // Grouping condition
  if (grouped === "true" && table) {
    where.id = {
      [Op.in]: model.sequelize.literal(
        `(SELECT MAX(id) FROM ${table} WHERE deleted = 0 GROUP BY name)`
      ),
    };
  }

  // Search condition for main model and includes
  if (search) {
    const searchConditions = [];

    // Search main model's text/string fields
    const mainModelFields = Object.keys(model.rawAttributes).filter((field) => {
      return (
        model.rawAttributes[field].type.key === "STRING" ||
        model.rawAttributes[field].type.key === "TEXT"
      );
    });

    mainModelFields.forEach((field) => {
      searchConditions.push({
        [`${table}.${field}`]: { [Op.like]: `%${search}%` },
      });
    });

    // Search fields from joined models if provided in 'includes'
    includes.forEach(({ as, model: includeModel, searchFields }) => {
      if (searchFields && searchFields.length > 0) {
        searchFields.forEach((field) => {
          searchConditions.push({
            [`${as}.${field}`]: { [Op.like]: `%${search}%` },
          });
        });
      }
    });

    where[Op.or] = searchConditions;
  }

  return where;
};

export const includeHelper = (relations) => {
  return relations.map(({ model, as, attributes, searchFields = [] }) => ({
    model,
    as,
    attributes,
    searchFields, // Pass searchable fields here
  }));
};

export const paginationHelper = (page, pageSize) => {
  const limit = pageSize ? parseInt(pageSize, 10) : 10;
  const offset = page ? (parseInt(page, 10) - 1) * limit : 0;
  return { limit, offset };
};

export const orderingHelper = (
  req,
  defaultField = "id",
  defaultDirection = "DESC"
) => {
  const orderField = req.query.orderField || defaultField;
  const orderDirection =
    req.query.orderDirection?.toUpperCase() === "ASC" ? "ASC" : "DESC";

  // If the order field includes a dot (.), it refers to a joined table's field
  const orderPath = orderField.includes(".")
    ? orderField.split(".")
    : [orderField];

  return [[...orderPath, orderDirection]];
};

/**Function to get comments on that policy */
export const getPolicyComment = async (policy_id) => {
  const [getPolicyComments] = await db.query(
    "SELECT policy_comments.created_by,policy_comments.created_at,policy_comments.status,policy_comments.comments,users.name,users.profile  FROM policy_comments left join users on users.id = policy_comments.created_by WHERE policy_comments.policy_id = ?",
    [policy_id]
  );
  // AND policy_comments.status = '0'
  return getPolicyComments;
};

export const getSopComment = async (sop_id) => {
  const [getSopComments] = await db.query(
    "SELECT sop_comments.created_by,sop_comments.created_at,sop_comments.status,sop_comments.comments,users.name,users.profile  FROM sop_comments left join users on users.id = sop_comments.created_by WHERE sop_comments.sop_id = ?",
    [sop_id]
  );
  // AND policy_comments.status = '0'
  return getSopComments;
};

export const getBcpComment = async (bcp_management_id) => {
  const [getBcpComments] = await db.query(
    "SELECT bcp_management_comment.created_by,bcp_management_comment.created_at,bcp_management_comment.status,bcp_management_comment.comments,users.name,users.profile  FROM bcp_management_comment left join users on users.id = bcp_management_comment.created_by WHERE bcp_management_comment.bcp_management_id = ?",
    [bcp_management_id]
  );
  return getBcpComments;
};

/**Function to get comments on that policy */
export const getKnowledgeComment = async (knowledge_recording_id) => {
  const [getKnowledgeComments] = await db.query(
    "SELECT knowledge_comment.reject_reason, knowledge_comment.created_by,knowledge_comment.created_at,knowledge_comment.status,knowledge_comment.comments,users.name,users.profile  FROM knowledge_comment left join users on users.id = knowledge_comment.created_by WHERE knowledge_comment.knowledge_recording_id = ?",
    [knowledge_recording_id]
  );
  return getKnowledgeComments;
};

export const getSupervisor = async (reviewerList) => {
  const [supervisor] = await db.query(
    "SELECT manager FROM users WHERE id IN (?)",
    [reviewerList]
  );
  return supervisor[0]?.manager;
};

export const getUserListByIds = async (ids) => {
  if (ids && ids.length && ids[0] !== "") {
    const userQuery = `SELECT users.id as user_id , CONCAT(users.name , ' ' , users.surname) as name , profile ,roles.id as role_id, roles.name as   role_name , 
          department.id as department_id, department.name as department_name from users
          LEFT JOIN roles ON roles.id = users.role 
          LEFT JOIN department ON department.id = users.department WHERE users.id In (${ids})
        `;
    const [userList] = await db.query(userQuery);
    return userList;
  }
};

export const getEmployeeByIds = async (ids) => {
  if (ids && ids.length && ids[0] !== "") {
    const userQuery = `SELECT users.id as employee ,
    users.unique_id as employee_unique_id,
    CONCAT(users.name , ' ' , users.surname) as employee_name , profile ,roles.id as role_id, roles.name as role, 
          department.id as department_id, department.name as department from users
          LEFT JOIN roles ON roles.id = users.role 
          LEFT JOIN department ON department.id = users.department WHERE users.id In (${ids})
        `;
    const [userList] = await db.query(userQuery);
    return userList;
  }
};

export const part_of_body_affected = [
  { id: 1, name: "Head and Face", parent_id: null },
  { id: 2, name: "Skull", parent_id: 1 },
  { id: 3, name: "Frontal bone (forehead)", parent_id: 2 },
  { id: 4, name: "Parietal bones (top and sides of the head)", parent_id: 2 },
  { id: 5, name: "Occipital bone (back of the head)", parent_id: 2 },
  {
    id: 6,
    name: "Temporal bones (sides of the head near the ears)",
    parent_id: 2,
  },
  { id: 7, name: "Sphenoid bone (behind the eyes)", parent_id: 2 },
  { id: 8, name: "Ethmoid bone (between the eyes)", parent_id: 2 },
  { id: 9, name: "Scalp", parent_id: 1 },
  { id: 10, name: "Skin", parent_id: 9 },
  { id: 11, name: "Hair follicles", parent_id: 9 },
  { id: 12, name: "Blood vessels", parent_id: 9 },
  { id: 13, name: "Face", parent_id: 1 },
  { id: 14, name: "Forehead", parent_id: 13 },
  { id: 15, name: "Eyes (Ocular region)", parent_id: 13 },
  { id: 16, name: "Eyelids", parent_id: 15 },
  { id: 17, name: "Cornea", parent_id: 15 },
  { id: 18, name: "Sclera", parent_id: 15 },
  { id: 19, name: "Lens", parent_id: 15 },
  { id: 20, name: "Retina", parent_id: 15 },
  { id: 21, name: "Optic nerve", parent_id: 15 },
  { id: 22, name: "Ears (Auditory system)", parent_id: 13 },
  { id: 23, name: "Outer ear (pinna)", parent_id: 22 },
  { id: 24, name: "Ear canal", parent_id: 22 },
  { id: 25, name: "Tympanic membrane (eardrum)", parent_id: 22 },
  {
    id: 26,
    name: "Middle ear (ossicles: malleus, incus, stapes)",
    parent_id: 22,
  },
  { id: 27, name: "Inner ear (cochlea, vestibular system)", parent_id: 22 },
  { id: 28, name: "Nose", parent_id: 13 },
  { id: 29, name: "Nasal bones", parent_id: 28 },
  { id: 30, name: "Nasal septum", parent_id: 28 },
  { id: 31, name: "Sinuses", parent_id: 28 },
  { id: 32, name: "Nostrils", parent_id: 28 },
  { id: 33, name: "Cheeks", parent_id: 13 },
  { id: 34, name: "Mouth", parent_id: 13 },
  { id: 35, name: "Lips", parent_id: 34 },
  { id: 36, name: "Teeth", parent_id: 34 },
  { id: 37, name: "Gums", parent_id: 34 },
  { id: 38, name: "Tongue", parent_id: 34 },
  { id: 39, name: "Palate (hard and soft)", parent_id: 34 },
  { id: 40, name: "Jaw (mandible, maxilla)", parent_id: 34 },
  { id: 41, name: "Salivary glands", parent_id: 34 },
  { id: 42, name: "Chin", parent_id: 34 },
  { id: 43, name: "Neck", parent_id: null },
  { id: 44, name: "Cervical Spine (vertebrae C1-C7)", parent_id: 43 },
  { id: 45, name: "Spinal Cord", parent_id: 43 },
  { id: 46, name: "Muscles", parent_id: 43 },
  { id: 47, name: "Sternocleidomastoid", parent_id: 46 },
  { id: 48, name: "Trapezius", parent_id: 46 },
  { id: 49, name: "Scalenes", parent_id: 46 },
  { id: 50, name: "Larynx (voice box)", parent_id: 43 },
  { id: 51, name: "Pharynx (throat)", parent_id: 43 },
  { id: 52, name: "Trachea (windpipe)", parent_id: 43 },
  { id: 53, name: "Esophagus", parent_id: 43 },
  { id: 54, name: "Carotid arteries and Jugular veins", parent_id: 43 },
  { id: 55, name: "Thyroid gland", parent_id: 43 },
  { id: 56, name: "Lymph nodes", parent_id: 43 },
  { id: 57, name: "Shoulders", parent_id: null },
  { id: 58, name: "Clavicle (collarbone)", parent_id: 57 },
  { id: 59, name: "Scapula (shoulder blade)", parent_id: 57 },
  { id: 60, name: "Humerus (upper arm bone)", parent_id: 57 },
  { id: 61, name: "Rotator cuff muscles", parent_id: 57 },
  { id: 62, name: "Supraspinatus", parent_id: 61 },
  { id: 63, name: "Infraspinatus", parent_id: 61 },
  { id: 64, name: "Teres minor", parent_id: 61 },
  { id: 65, name: "Subscapularis", parent_id: 61 },
  { id: 66, name: "Acromioclavicular (AC) joint", parent_id: 57 },
  { id: 67, name: "Glenohumeral joint (ball-and-socket joint)", parent_id: 57 },
  { id: 68, name: "Deltoid muscle", parent_id: 57 },
  { id: 69, name: "Ligaments", parent_id: 57 },
  { id: 70, name: "Coracoacromial ligament", parent_id: 69 },
  { id: 71, name: "Chest/Thorax", parent_id: null },
  { id: 72, name: "Rib cage", parent_id: 71 },
  { id: 73, name: "Ribs (1-12)", parent_id: 72 },
  { id: 74, name: "Costal cartilage", parent_id: 72 },
  { id: 75, name: "Sternum (breastbone)", parent_id: 72 },
  { id: 76, name: "Lungs", parent_id: 71 },
  { id: 77, name: "Heart", parent_id: 71 },
  { id: 78, name: "Diaphragm", parent_id: 71 },
  { id: 79, name: "Intercostal muscles", parent_id: 71 },
  { id: 80, name: "Thoracic vertebrae (T1-T12)", parent_id: 71 },
  { id: 81, name: "Abdomen", parent_id: null },
  { id: 82, name: "Stomach", parent_id: 81 },
  { id: 83, name: "Liver", parent_id: 81 },
  { id: 84, name: "Gallbladder", parent_id: 81 },
  { id: 85, name: "Pancreas", parent_id: 81 },
  { id: 86, name: "Spleen", parent_id: 81 },
  { id: 87, name: "Small intestine", parent_id: 81 },
  { id: 88, name: "Large intestine", parent_id: 81 },
  { id: 89, name: "Kidneys", parent_id: 81 },
  { id: 90, name: "Bladder", parent_id: 81 },
  { id: 91, name: "Pelvis", parent_id: null },
  { id: 92, name: "Hip bones (ilium, ischium, pubis)", parent_id: 91 },
  { id: 93, name: "Sacrum", parent_id: 91 },
  { id: 94, name: "Coccyx (tailbone)", parent_id: 91 },
  { id: 95, name: "Reproductive organs", parent_id: 91 },
  { id: 96, name: "Muscles", parent_id: 91 },
  { id: 97, name: "Legs", parent_id: null },
  { id: 98, name: "Thigh (femur)", parent_id: 97 },
  { id: 99, name: "Knee", parent_id: 97 },
  { id: 100, name: "Lower leg (tibia, fibula)", parent_id: 97 },
  { id: 101, name: "Ankle", parent_id: 97 },
  { id: 102, name: "Foot", parent_id: 97 },
  { id: 103, name: "Arms", parent_id: null },
  { id: 104, name: "Upper arm (humerus)", parent_id: 103 },
  { id: 105, name: "Elbow", parent_id: 103 },
  { id: 106, name: "Forearm (radius, ulna)", parent_id: 103 },
  { id: 107, name: "Wrist", parent_id: 103 },
  { id: 108, name: "Hand", parent_id: 103 },
];

export const getEmployeeDetails = async (
  bcpManagementData,
  crisisManagementData = null
) => {
  const allEmployeeFields = [
    "crisis_management_team_leader", // Leader field
    "deputy_crisis_management_team_leader",
    "emergency_response_coordinator",
    "technical_recovery_lead",
    "communication_officer",
    "legal_and_compliance_officer",
    "hr_representative",
    "operation_recovery_lead",
    "finance_and_administration_lead",
    "facilities_and_physical_infrastructure",
    "supply_chain_and_procurement_lead",
    "external_liaison_officer",
    "security_lead",
    "risk_management_officer",
    "crisis_documentation_officer",
  ];

  const employeeFieldIds = [];
  const employeeFieldMap = {};

  // Collect all user IDs for batch query
  for (let item of bcpManagementData) {
    for (let key of allEmployeeFields) {
      if (item[key]) {
        employeeFieldIds.push(item[key]);
        employeeFieldMap[item[key]] = { key, id: item[key] };
      }
    }
  }

  if (employeeFieldIds.length > 0) {
    const uniqueIds = [...new Set(employeeFieldIds)];

    const leaderQuery = `
        SELECT 
          users.id AS user_id,
          CONCAT(users.name, ' ', users.surname) AS name,
          users.profile AS profile,
          users.phone AS phone_number,
          department.name AS department,
          roles.name AS role,
          users.email AS email_address 
        FROM users 
        LEFT JOIN roles ON roles.id = users.role 
        LEFT JOIN department ON department.id = users.department 
        WHERE users.id IN (${uniqueIds.join(",")})
      `;

    const [leaderData] = await db.query(leaderQuery);

    // Map user data by user_id
    const leaderDataMap = leaderData.reduce((map, leader) => {
      map[leader.user_id] = leader;
      return map;
    }, {});

    // Prepare all_members array for return
    const all_members = leaderData.map((leader) => ({
      value: leader.user_id,
      title: leader.name,
    }));

    // Merge leader and member data into the bcpManagementData
    for (let i = 0; i < bcpManagementData.length; i++) {
      const item = bcpManagementData[i];
      let all_members_for_item = [];
      let leader_for_item = null;

      for (let key of allEmployeeFields) {
        if (item[key] && leaderDataMap[item[key]]) {
          const leader = leaderDataMap[item[key]];
          const memberDetails = {
            [`${key}_name`]: leader.name || null,
            [`${key}_phone_number`]: leader.phone_number || null,
            [`${key}_department`]: leader.department || null,
            [`${key}_role`]: leader.role || null,
            [`${key}_email_address`]: leader.email_address || null,
            [`${key}_profile`]: leader.profile || null,
          };
          // Add member details to the current BCP management item
          bcpManagementData[i] = {
            ...bcpManagementData[i],
            ...memberDetails,
          };

          // Add to all members
          all_members_for_item.push({
            value: leader.user_id,
            title: leader.name,
          });

          // Check if this is the "crisis_management_team_leader" (Leader)
          if (key === "crisis_management_team_leader") {
            leader_for_item = {
              value: leader.user_id,
              title: leader.name,
              phone_number: leader.phone_number || null,
              email_address: leader.email_address || null,
              department: leader.department || null,
              profile: leader.profile || null,
              role: leader.role || null,
            };
          }
        }
      }

      // Add the leader and all_members to the target object
      if (crisisManagementData) {
        crisisManagementData[i] = crisisManagementData[i] || {};
        crisisManagementData[i].cmt_leader_name = leader_for_item?.title; // Add leader data
        crisisManagementData[i].all_members = all_members_for_item; // Add all members
      } else {
        bcpManagementData[i].cmt_leader_name = leader_for_item?.title; // Add leader data
        bcpManagementData[i].all_members = all_members_for_item; // Add all members
      }
    }

    // Return the updated data along with all_members and leaderDataMap
    return { leaderDataMap, all_members, bcpManagementData };
  }

  return { leaderDataMap: {}, all_members: [], bcpManagementData };
};

export const auditLogFunction = async (data) => {
  try {
    data.unique_id = await uniqueIdGenerator(
      data?.organization,
      data?.department,
      "LOG",
      "audit_log",
      "unique_id",
      "unique_id"
    );
    const { query, values } = createQueryBuilder(AuditLog, data);
    await db.query(query, values);
    return true;
  } catch (error) {
    storeError(error);
    return;
  }
};

const data = {
  repository_id: 159,
  created_by: 1,
  document_name: "first document",
  action_type: "edit",
  document_status: "In Progress",
};

// console.log(await auditLogFunction(data));

export const updateArchiveRepoDoc = async (id) => {
  if (!id) {
    throw new Error("id is required");
  }
  await db.query(
    `UPDATE repository SET is_archived = 1 where deleted = 0 AND id = ?`,
    [id]
  );
  return true;
};

// export const processUploadDocuments = async (req, sidebar_id) => {
//   const documents = [];

//   for (let i = 0; req.body[`upload_documents[${i}][title]`] !== undefined; i++) {
//     const id = i + 1;
//     const title = req.body[`upload_documents[${i}][title]`];
//     const file = req.files && req.files[`upload_documents[${i}][file]`];
//     let ddrm_id = "";

//     if (typeof file !== "string" && typeof file !== "undefined" && file !== null) {
//       // Upload the file and get the DDRM ID
//       ddrm_id = await uploadToDDRM(sidebar_id, file, req);

//       // Archive the old document if an old DDRM ID exists
//       const old_ddrm_id = req.body[`upload_documents[${i}][ddrm_id]`];
//       if (old_ddrm_id) {
//         await updateArchiveRepoDoc(old_ddrm_id);
//       }
//     } else {
//       // Use existing DDRM ID if no new file is provided
//       ddrm_id = req.body[`upload_documents[${i}][ddrm_id]`];
//     }

//     // Add processed document to the array
//     documents.push({ id, title, ddrm_id });
//   }

//   return documents;
// };
export const employeeCheck = [
  "background_check",
  "identity_verification",
  "skills_verification",
  "aptitude_and_reasoning_tests",
  "probation_period_evaluation",
  "continuous_training_and_development",
  "psychometric_personality_assessment",
  "medical_and_physical_tests",
  "financial_and_legal_checks",
];

export const pre_employment_check = {
  background_check: [
    { name: "Criminal Background Check", completed: false, file: null },
    { name: "Credit Check", completed: false, file: null },
    { name: "Employment History Verification", completed: false, file: null },
    { name: "Education Verification", completed: false, file: null },
    {
      name: "Professional License Verification",
      completed: false,
      file: null,
    },
    { name: "Reference Checks", completed: false, file: null },
    { name: "Social Media Check", completed: false, file: null },
  ],
  identity_verification: [
    {
      name: "Government Issued ID Verification",
      completed: false,
      file: null,
    },
    { name: "Work Authorization Check", completed: false, file: null },
  ],
  skills_verification: [
    {
      name: "Job-Specific Test",
      completed: false,
      file: null,
    },
    { name: "Language Proficiency Test", completed: false, file: null },
    { name: "Typing Test", completed: false, file: null },
  ],
  psychometric_personality_assessment: [
    {
      name: "Verbal Reasoning Test",
      completed: false,
      file: null,
    },
    { name: "Emotional Intelligence Test", completed: false, file: null },
    { name: "Cognitive Ability Test", completed: false, file: null },
    { name: "Behavioural Test", completed: false, file: null },
  ],
  aptitude_and_reasoning_tests: [
    {
      name: "Numerical Reasoning Test",
      completed: false,
      file: null,
    },
    { name: "Logical/Abstract Reasoning Test", completed: false, file: null },
    { name: "Typing Test", completed: false, file: null },
  ],
  medical_and_physical_tests: [
    {
      name: "Pre-Employment Medical Examination",
      completed: false,
      file: null,
    },
    { name: "Drug Screening Test", completed: false, file: null },
    { name: "Vision and Hearing Test", completed: false, file: null },
    { name: "Physical Fitness Test", completed: false, file: null },
  ],
  financial_and_legal_checks: [
    {
      name: "Non-Disclosure Agreement (NDA) Compliance",
      completed: false,
      file: null,
    },
    {
      name: "Non-Compete/Non-Solicitation Agreement Review",
      completed: false,
      file: null,
    },
  ],
};

export const post_employment_check = {
  probation_period_evaluation: [
    { name: "Performance Review", completed: false, file: null },
    { name: "Peer Feedback", completed: false, file: null },
    { name: "Goal Achievement Monitoring", completed: false, file: null },
  ],
  continuous_training_and_development: [
    {
      name: "Skills Assessment",
      completed: false,
      file: null,
    },
    { name: "Training and Development Reviews", completed: false, file: null },
    { name: "Certification Renewal Tracking", completed: false, file: null },
  ],
};
export const processUploadDocuments = async (
  req,
  sidebar_id,
  key = "upload_documents"
) => {
  const documents = [];

  //   console.log("req.body",req.body)
  //   console.log("req.files",typeof req.body)

  //   if (typeof req.body === 'string') {
  //     req.body = JSON.parse(req.body);
  //   }

  //   console.log("-----------------")
  //   console.log("->>>>>>>>>>",req.body.upload_documents[1])

  //   console.log("Type of upload_documents[0]:", typeof req.body.upload_documents[0]);
  // console.log("Type of upload_documents[1]:", typeof req.body.upload_documents[1]);
  // console.log("Full upload_documents:", req.body.upload_documents);

  // req.body[key] = JSON.parse(req.body[key])

  for (
    let i = 0;
    // req.body[`${key}[${i}][title]`] !== undefined ||
    // req.body[`${key}[${i}][ddrm_id]`] !== undefined ||
    // req.body[key]?.[i]?.ddrm_id ||
    // req.body?.[key]?.[i]?.ddrm_id !== undefined ||
    // req.body[key]?.[i]?.ddrm_id !== undefined ||
    // req.body?.[key]?.[i]?.["completed"];
    i < req.body[key]?.length;
    i++
  ) {
    // console.log("req.body?.[key][i]?.ddrm_id: ", req.body?.[key][i]?.ddrm_id);
    const document = {}; // Dynamically collect all fields for this document
    const file = req.files && req.files[`${key}[${i}][file]`];
    console.log('file: ', file);

    // console.log((">>>>>>>>>>>>",file))
    const completed = req.body?.[key]?.[i]?.["completed"];
    // console.log("completed: ", completed);
    // Dynamically add all known fields to the document object
    if (req.body[`${key}[${i}][title]`] !== undefined) {
      document.title = req.body[`${key}[${i}][title]`];
    }
    // if (completed) {
    document.name = req.body?.[key]?.[i]?.["name"];
    document.completed = req.body?.[key]?.[i]?.["completed"];
    document.id = req.body?.[key]?.[i]?.["id"];
    // console.log(
    //   'req.body?.[key]?.[i]?.["completed"]: ',
    //   req.body?.[key]?.[i]?.["completed"]
    // );
    document.key = key;
    delete req.body[`${key}`][i]["file"];

    // }

    if (req.body[`${key}[${i}][classification]`] !== undefined) {
      document.classification = req.body[`${key}[${i}][classification]`];
    }
    if (req.body[`${key}[${i}][file_classification]`] !== undefined) {
      document.file_classification =
        req.body[`${key}[${i}][file_classification]`];
    }

    let ddrm_id = "";

    // If the file is provided, upload it; otherwise, use the existing `ddrm_id`
    if (
      typeof file !== "string" &&
      typeof file !== "undefined" &&
      file !== null
    ) {
      ddrm_id = await uploadToDDRM(sidebar_id, file, req);

      // Archive the old document if an old DDRM ID exists
      // const old_ddrm_id = req.body[`${key}[${i}][ddrm_id]`];
      const old_ddrm_id = req.body[key][i]?.ddrm_id;
      if (old_ddrm_id) {
        await updateArchiveRepoDoc(old_ddrm_id);
      }
    } else {
      // ddrm_id = req.body[`${key}[${i}][ddrm_id]`];
      ddrm_id = req.body[key] && req.body[key][i]?.ddrm_id;
    }

    document.ddrm_id = ddrm_id;

    // console.log("++++++++++",document)

    // Add the processed document to the documents array
    documents.push(document);
  }

  return documents;
};

export const processesSingleDDRMDocument = async (
  table_name,
  sidebar_id,
  file,
  req
) => {
  const { id } = req.body;
  if (id) {
    const [record] = await getRecord(table_name, "id", id);
    const updateQuery = `UPDATE repository set is_archived = 1 where id = ${record?.ddrm_id}`;
    await db.query(updateQuery);
  }
  const ddrm_id = await uploadToDDRM(sidebar_id, file, req);
  return ddrm_id;
};

/**
 * Recursively calculates the total size of a folder, including nested files and subfolders.
 * @param {string} type - The type of the item, e.g., 'folder'.
 * @param {string} url - The absolute path of the folder.
 * @returns {number} - The total size of the folder in bytes.
 */
export const getFolderTotalSizeRecursively = (type, url) => {
  let totalSize = 0;

  try {
    // console.log('url: ', url);
    let joinPublicPath;
    if (!url.includes("public")) {
      joinPublicPath = path.join(__dirname, `../public/${url}`);
    } else joinPublicPath = url;

    const normalizedUrl = path.resolve(joinPublicPath);
    // const normalizedUrl = path.resolve(url);
    if (type === "folder" && fs.existsSync(normalizedUrl)) {
      // Read the contents of the folder
      const items = fs.readdirSync(normalizedUrl);

      for (const item of items) {
        const itemPath = path.join(normalizedUrl, item);
        const itemStats = fs.statSync(itemPath);

        if (itemStats.isFile()) {
          // Add the size of the file
          totalSize += itemStats.size;
        } else if (itemStats.isDirectory()) {
          // Recursively calculate the size of the subfolder
          totalSize += getFolderTotalSizeRecursively(type, itemPath);
        }
      }
      return totalSize;
    }
  } catch (error) {
    throw error;
  }
};

export function getUniqueFolderName(siblings, folderName) {
  let counter = 0;
  let uniqueName = folderName;

  // Check if folder name already exists
  const siblingNames = siblings.map((sibling) => sibling.name);

  while (siblingNames.includes(uniqueName)) {
    counter++;
    uniqueName = `${folderName}(${counter})`;
  }

  return uniqueName;
}

export async function saveEquipmentHistory(
  // tableName,
  action,
  department,
  organization,
  customMessage = "",
  createdBy,
  equipment_id,
  maintenance_inspection_id = null,
  changedFields = null,
  last_updated_date = null,
  remarks_comments = null,
  reported_by = null,
  notify_to = null
) {
  try {
    // const query = `
    //       INSERT INTO fire_equipment_status_history ( equipment_id, action,department,organization,created_by, maintenance_inspection_id,changed_fields, message, updated_date)
    //       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    //   `;
    // await db.query(query, [
    //   equipment_id,
    //   action,
    //   department,
    //   organization,
    //   createdBy,
    //   maintenance_inspection_id,
    //   changedFields ? JSON.stringify(changedFields) : null,
    //   customMessage,
    //   moment().format("YYYY-MM-DD"),
    // ]);

    const { query: HistoryQuery, values: HistoryValues } = createQueryBuilder(
      FireEquipmentStatusHistory,
      {
        equipment_id,
        updated_date: last_updated_date
          ? last_updated_date
          : moment().format("YYYY-MM-DD"),
        comments: remarks_comments,
        reported_by,
        notify_to,
        maintenance_inspection_id,
        created_by: createdBy,
        action,
        changed_fields: changedFields ? JSON.stringify(changedFields) : null,
        message: customMessage,
        department,
        organization,
      }
    );

    await db.query(HistoryQuery, HistoryValues);
  } catch (error) {
    throw error;
  }
}

export async function copyRecursiveAsync(src, dest) {
  try {
    const fsPromises = fs.promises;

    const stats = await fsPromises.lstat(src);

    if (stats.isDirectory()) {
      // Create the destination directory if it doesn't exist
      await fsPromises.mkdir(dest, { recursive: true });

      // Read the contents of the directory
      const items = await fsPromises.readdir(src);

      // Recursively copy each item in the directory
      await Promise.all(
        items.map(async (item) => {
          const srcItem = path.join(src, item);
          const destItem = path.join(dest, item);
          await copyRecursiveAsync(srcItem, destItem);
        })
      );
    } else if (stats.isFile()) {
      // Copy the file
      await fsPromises.copyFile(src, dest);
    }
  } catch (error) {
    throw error;
  }
}

export const incidentCategoryConstant = {
  Safety: [
    {
      name: "Injury On Duty Technical Form",
      tableName: "injury_on_duty",
      key: "injured_persons",
      structureData: {
        user_id: "",
        injury_classification: "",
        parts_of_body_affected: [],
        reportable_classification: "",
        reported_authorities: "No",
        reported_sphere: "",
        reported_name: "",
        reported_sector: "",
        regulator_name: "",
        date_reported: "",
        reported_by: "",
        hospitalized: "No",
        hospital__state: "",
        hospital_name: "",
      },
    },
    {
      name: "Explosion Bomb Threat Technical Form",
      tableName: "explosion_bomb_form",
      key: "explosion_bomb_form",
    },
    {
      name: "HAZMAT and Fire Technical Form",
      tableName: "hazmat_fire_technical_form",
      key: "hazmat_fire_technical_form",
    },
  ],
  Health: [
    {
      name: "Occupational Disease Technical Form",
      tableName: "occupational_disease_form",
      key: "occupationals",
    },
  ],
  Environment: [
    {
      name: "Environmental Incident Technical Form",
      tableName: "environmental_incidents",
      key: "incident_type",
    },
  ],
  Security: [
    {
      name: "Security Incidents Technical Form",
      tableName: "security_incident_form",
    },
  ],
  Assets: [
    {
      name: "Vehicle Incident Technical Form",
      tableName: "vehicle_incident_form",
      key: "injured_persons",
    },
    {
      name: "Assets Incident Technical Form",
      tableName: "infrastructure_technical_form",
      key: "asset_type",
    },
  ],
  Quality: [
    {
      name: "Non Compliance Technical Form",
      tableName: "non_compliance_technical_form",
      key: "quality_incident_details",
      structureData: {
        quality_incident_type: "",
        quality_incident_data: {
          quantity_affected: "",
          value: "",
          currency: "",
          customer_impact: "",
          actual_value: "",
          actual_currency: "",
        },
        customer_complaints_details: {
          value: "",
          currency: "",
          actual_value: "",
          actual_currency: "",
        },
        process_deviation_incident: {
          value: "",
          currency: "",
          customer_impact: "",
          actual_value: "",
          actual_currency: "",
        },
        suplier_quality_details: {
          financial_value: "",
          financial_currency: "",
          customer_impact: "",
          actual_value: "",
          actual_currency: "",
        },
        test_inspection_faliures: {
          financial_value: "",
          financial_currency: "",
          customer_impact: "",
          actual_value: "",
          actual_currency: "",
        },
        documentations_error: {
          product_impact: "",
          value: "",
          currency: "",
          customer_impact: "",
          actual_value: "",
          actual_currency: "",
        },
        rework_and_repair: {
          value: "",
          currency: "",
          actual_value: "",
          actual_currency: "",
        },
        change_control_failures: {
          financial_value: "",
          financial_currency: "",
          actual_value: "",
          actual_currency: "",
        },
        it_and_system_failure: {
          value: "",
          currency: "",
          actual_value: "",
          actual_currency: "",
        },
        supply_chain_disruption: {
          financial_value: "",
          financial_currency: "",
          actual_value: "",
          actual_currency: "",
        },
        trainig_and_competency: {
          value: "",
          currency: "",
          actual_value: "",
          actual_currency: "",
        },
        product_recall_details: {
          value: "",
          currency: "",
          actual_value: "",
          actual_currency: "",
        },
        calibration_and_maintenance: {
          value: "",
          currency: "",
          customer_impact: "",
          actual_value: "",
          actual_currency: "",
        },
      },
    },
  ],
  "Near Miss": [
    { name: "Near Miss Reporting", tableName: "near_miss_reporting" },
  ],
};

/**
 *
 * @param {Number} incidentId
 * @returns {Object}
 */
export const getIncidentInvestigationData = async (incidentId) => {
  try {
    const pageSize = 10;
    const page = 1;
    const all = true;

    const incidentRecord = await getRecord("incident", "id", incidentId);
    if (incidentRecord.length === 0)
      return { status: 404, message: "Report Not Found" };

    const incident = incidentRecord[0];
    incident.incident_category = JSON.parse(
      incident?.incident_category || "[]"
    );

    // Fetch selected incident category list
    const selectedCategoryQuery = `SELECT id, name FROM incident_category WHERE deleted = 0 AND id IN (${incident.incident_category})`;
    let [selectedCategoryList] = await db.query(selectedCategoryQuery);

    // Fetch all incident categories
    const incidentCategoryQuery = `SELECT MAX(id) as id, name FROM incident_category WHERE deleted = 0 GROUP BY name`;
    const [incidentCategoryList] = await db.query(incidentCategoryQuery);

    let result = [];
    let incidentCategoryCheckObj = {};

    for (let i = 0; i < incidentCategoryList.length; i++) {
      const item = incidentCategoryList[i];

      if (incidentCategoryConstant[item.name]) {
        const incidentCategoryFormList = incidentCategoryConstant[item.name];

        for (let j = 0; j < incidentCategoryFormList.length; j++) {
          const e = incidentCategoryFormList[j];

          if (e.name && !incidentCategoryCheckObj[item.name]) {
            incidentCategoryCheckObj[item.name] = true;

            let status = "Incomplete";
            let form_id = "";
            let form_data = {};

            if (e.tableName) {
              const FormCheckQuery = `SELECT * FROM ${e.tableName} WHERE incident_id = ${incidentId} AND deleted = 0 ORDER BY id DESC`;
              let [formRecord] = await db.query(FormCheckQuery);
              formRecord = await decodeAndParseFields(formRecord);

              if (formRecord.length > 0) {
                status = "Completed";
                form_id = formRecord[0]?.id;
                form_data[e.key] = formRecord[0][e.key] || {};

                if (!e.key) {
                  form_data = formRecord[0];
                }

                // Handle Assets case
                if (item.name === "Assets") {
                  const assetsQuery = `SELECT * FROM infrastructure_technical_form WHERE incident_id = ${incidentId} AND deleted = 0 ORDER BY id DESC`;
                  let [assetsRecord] = await db.query(assetsQuery);
                  assetsRecord = await decodeAndParseFields(assetsRecord);

                  if (assetsRecord.length > 0) {
                    form_data.asset_type = assetsRecord[0]["asset_type"];
                  }
                }
              }

              if (formRecord[0]?.save_type === "draft") status = "Draft";
            }

            result.push({
              ...item,
              title: e.name,
              status,
              form_id,
              incident_id: incidentId,
              form_data,
            });
          }
        }
      }
    }

    result = result.map((e) => {
      const categorySelectedCheck = selectedCategoryList.find(
        (item) => item.name === e.name
      );

      return categorySelectedCheck ? e : { ...e, status: "Not Added" };
    });

    const totalRecord = result.length;
    let recordList = result;

    // Apply manual pagination if all is false
    if (!all) {
      recordList = result.slice((page - 1) * pageSize, page * pageSize);
    }

    return {
      status: 200,
      data: {
        records: recordList,
        totalRecord,
      },
    };
  } catch (error) {
    console.error("Error fetching incident investigation data:", error);
    return { status: 500, message: "Internal Server Error" };
  }
};

export const createUpdateAction = async (data, organization, sessionId) => {
  try {
    if (typeof data === "string") {
      data = JSON.parse(data);
    }
    // let organization;
    // if (department) {
    //   organization = (await getOrganizationAccordingToDepartment(department))[0].organization;
    // }
    const arr = [];
    for (let action of data) {
      if (action?.necessary == 1) {
        action[action?.id ? "updated_by" : "created_by"] = sessionId;
        action.organization = organization;
        action.task_title = action.action;
        action.responsible_person = action.who;
        const startDate = new Date(action.when);
        const endDate = new Date(startDate.setDate(startDate.getDate() + 1));

        const differenceInMilliseconds = endDate - startDate;

        const differenceInDays =
          differenceInMilliseconds / (1000 * 60 * 60 * 24);

        const totalWorkingHours = differenceInDays * 8;
        action.estimated_time = totalWorkingHours;
        action = await encodeAndStringifyFields(action);
        const { query, values } = action.id
          ? updateQueryBuilder(CustomActionCreation, action)
          : createQueryBuilder(CustomActionCreation, action);
        // console.log(query);
        const [result] = await db.query(query, values);

        if (!action?.id) {
          action.id = result.insertId;
        }
        arr.push(action);
      } else {
        arr.push(action);
      }
    }
    return arr;
  } catch (error) {
    throw error;
  }
};

export const getQualityFormData = async (data) => {
  const typeToKeyMap = {
    "Defective Product Incidents": "quality_incident_data",
    "Customer Complaints": "customer_complaints_details",
    "Process Deviation Incidents": "process_deviation_incident",
    "Supplier Quality Incidents": "suplier_quality_details",
    "Documentation Errors": "documentations_error",
    "Test and Inspection Failures": "test_inspection_faliures",
    "Change Control Failures": "change_control_failures",
    "Rework and Repair Incidents": "rework_and_repair",
    "IT and System Failures": "it_and_system_failure",
    "Calibration and Maintenance Failures": "calibration_and_maintenance",
    "Product Recall Incidents": "product_recall_details",
    "Training and Competency Incidents": "trainig_and_competency",
    "Supply Chain Disruptions": "supply_chain_disruption",
  };
  try {
    for (let item of data) {
      const incidentType = item.quality_incident_type;
      const key = typeToKeyMap[incidentType];
      console.log("key: ", key);

      if (key && item[key]) {
        const {
          currency,
          actual_currency,
          financial_currency,
          notify,
          technical_employee_name,
          ddrm_id,
          file_classification,
          employee_name,
          defect_detection_method,
          defect_type,
          personnel_involved,
          document_owner,
          shift_info,
          process_owner,
          deviation_category,
          is_contractor,
          contractor_name,
          project_id,
          personnel_affected,
          teams_affected,
          department_affected,
        } = item[key];

        if (currency) {
          const [currencyRows] = await db.query(
            "SELECT name FROM currency WHERE id = ?",
            [currency]
          );
          item[key].currency_name = currencyRows[0]?.name;
        }
        if (actual_currency) {
          const [currencyRows] = await db.query(
            "SELECT name FROM currency WHERE id = ?",
            [actual_currency]
          );
          item[key].actual_currency_name = currencyRows[0]?.name;
        }
        if (financial_currency) {
          const [currencyRows] = await db.query(
            "SELECT name FROM currency WHERE id = ?",
            [financial_currency]
          );
          item[key].financial_currency_name = currencyRows[0]?.name;
        }

        if (notify && notify.length > 0 && notify[0].employee_name) {
          const userIds = notify.map((n) => n.employee_name);
          const [userRows] =
            await db.query(`SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id IN(${userIds})`);

          item[key].notify = userRows;
        }
        if (
          personnel_affected &&
          personnel_affected.length > 0 &&
          personnel_affected[0].employee_name
        ) {
          const userIds = personnel_affected.map((n) => n.employee_name);
          const [userRows] =
            await db.query(`SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id IN(${userIds})`);

          item[key].personnel_affected = userRows;
        }
        if (
          teams_affected &&
          teams_affected.length > 0 &&
          teams_affected[0].teams_affected_name
        ) {
          const teamIds = teams_affected.map((n) => n.teams_affected_name);
          const [teamDetails] = await db.query(
            `SELECT id as teams_affected_id,name as teams_affected_name FROM teams WHERE id IN(${teamIds})`
          );

          item[key].teams_affected = teamDetails;
        }
        if (
          department_affected &&
          department_affected.length > 0 &&
          department_affected[0].department_affected_name
        ) {
          const departmentIds = department_affected.map(
            (n) => n.department_affected_name
          );
          const [departmentDetails] = await db.query(
            `SELECT id as department_affected_id,name as department_affected_name FROM department WHERE id IN(${departmentIds})`
          );

          item[key].department_affected = departmentDetails;
        }
        if (
          personnel_involved &&
          key === "rework_and_repair" &&
          personnel_involved.length > 0 &&
          personnel_involved[0].employee_name
        ) {
          const userIds = personnel_involved.map((n) => n.employee_name);
          const [userRows] =
            await db.query(`SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id IN(${userIds})`);

          item[key].personnel_involved = userRows;
        }

        if (employee_name) {
          const [user] = await db.query(
            `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
            [employee_name]
          );

          item[key].employee_id = user[0].user_id;
          item[key].employee_name = user[0].name;
          item[key].profile = user[0].profile;
          item[key].department = user[0].department;
          item[key].role = user[0].role;
        }
        if (technical_employee_name) {
          const [user] = await db.query(
            `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
            [technical_employee_name]
          );

          item[key].technical_employee_id = user[0]?.user_id;
          item[key].technical_employee_name = user[0]?.name;
          item[key].profile = user[0]?.profile;
          item[key].technical_department = user[0]?.department;
          item[key].technical_role = user[0]?.role;
        }

        if (ddrm_id) {
          const [ddrmData] = await db.query(
            "SELECT repository.url as file,document_name as title FROM repository LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE repository.deleted = 0 AND id = ?",
            [ddrm_id]
          );

          item[key].file = ddrmData[0].file;
          item[key].title = ddrmData[0].title;
        }

        if (file_classification) {
          const [fileClassificationData] = await db.query(
            "SELECT name FROM file_classification WHERE id = ?",
            [file_classification]
          );
          item[key].file_classification_name = fileClassificationData[0].name;
        }

        if (defect_detection_method) {
          const [defectDetectionMethodData] = await db.query(
            "SELECT name FROM defect_detection_method WHERE id = ?",
            [defect_detection_method]
          );
          item[key].defect_detection_method_name =
            defectDetectionMethodData[0].name;
        }

        if (defect_type) {
          const [defectTypeData] = await db.query(
            "SELECT name FROM defect_type WHERE id = ?",
            [defect_type]
          );
          item[key].defect_type_name = defectTypeData[0].name;
        }

        if (document_owner) {
          const [user] = await db.query(
            `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
            [document_owner]
          );

          item[key].document_owner = user[0].user_id;
          item[key].document_owner_name = user[0].name;
          item[key].document_owner_profile = user[0].profile;
        }

        if (shift_info) {
          const [shiftData] = await db.query(
            "SELECT name FROM shift WHERE id = ?",
            [shift_info]
          );
          item[key].shift_name = shiftData[0].name;
        }

        if (process_owner) {
          const [department] = await db.query(
            "SELECT name FROM department WHERE id = ?",
            [process_owner]
          );
          item[key].process_owner_name = department[0].name;
        }
        if (deviation_category) {
          const [category] = await db.query(
            "SELECT name FROM category WHERE id = ?",
            [deviation_category]
          );
          item[key].deviation_category_name = category[0].name;
        }

        if (
          key === "process_deviation_incident" &&
          personnel_involved === "External" &&
          is_contractor == 1
        ) {
          const [contractor] = await db.query(
            "SELECT id, contractor_name FROM contractor_registration WHERE id = ?",
            [contractor_name]
          );
          item[key].contractor_id = contractor[0].id;
          item[key].contractor_name = contractor[0].contractor_name;
          const [project] = await db.query(
            "SELECT id, project_title FROM project_registration WHERE id = ?",
            [project_id]
          );
          item[key].project_id = project[0].id;
          item[key].project_name = project[0].project_title;
        }
      }
    }
    return data;
  } catch (error) {
    throw error;
  }
};

export const getEnvironmentIncidentTypeData = async (data) => {
  try {
    for (let incident of data) {
      const { severity, reported_by, waste_description } = incident;

      if (severity) {
        const [severityData] = await db.query(
          "SELECT name FROM severity WHERE id = ?",
          [severity]
        );
        incident.severity_name = severityData[0].name;
      }

      if (reported_by) {
        const [user] = await db.query(
          "SELECT CONCAT(users.name , ' ' , users.surname) as name FROM users WHERE id ?",
          [reported_by]
        );
        incident.reported_by_name = user[0].name;
      }
      if (waste_description) {
        const [wasteData] = await db.query(
          "SELECT name FROM waste_description WHERE id = ?",
          [waste_description]
        );
        incident.waste_description_name = wasteData[0].name;
      }
    }
    return data;
  } catch (error) {
    throw error;
  }
};

export const getCrisisManagementJSONData = async (rows) => {
  const processedData = await Promise.all(
    rows.map(async (row) => {
      // Handle crisis_response_actions
      const crisisResponseActions = await Promise.all(
        (row.crisis_response_actions || []).map(async (action) => {
          const [user] = await db.query(
            "SELECT CONCAT(users.name, ' ', users.surname) as name, users.profile FROM users WHERE id = ?",
            [action.responsible_person]
          );
          return {
            immediate_response_actions: action.immediate_response_actions,
            responsible_person: action.responsible_person,
            responsible_person_name: user?.[0]?.name,
            responsible_person_profile: user?.[0]?.profile,
          };
        })
      );

      // Handle immediate_decisions_made
      const immediateDecisionsMade = await Promise.all(
        (row.immediate_decisions_made || []).map(async (decision) => {
          const [user] = await db.query(
            "SELECT users.name, users.surname, users.profile FROM users WHERE id = ?",
            [decision.responsible_person]
          );
          return {
            decision_date: decision.decision_date,
            decision_time: decision.decision_time,
            description: decision.description,
            responsible_person: decision.responsible_person,
            responsible_person_name: user?.[0]?.name || "Unknown",
            responsible_person_surname: user?.[0]?.surname || "Unknown",
            responsible_person_profile: user?.[0]?.profile || "Unknown",
          };
        })
      );

      // Handle crisis_communications
      const crisisCommunications = row.crisis_communications || {};
      const communicationDetails = await Promise.all(
        (crisisCommunications.communication_details || []).map(
          async (detail) => {
            const [method] = await db.query(
              "SELECT name FROM communication_method WHERE id = ?",
              [detail.communication_method]
            );
            return {
              communication_method: detail.communication_method,
              communication_method_name: method?.[0]?.name,
              communication_date: detail.communication_date,
              communication_time: detail.communication_time,
              communication_content: detail.communication_content,
            };
          }
        )
      );

      // Handle resource_allocation
      const resourceAllocation = row.resource_allocation || {};
      const humanResources = await Promise.all(
        (resourceAllocation.human_resources || []).map(async (resource) => {
          if (resource?.role) {
            const [role] = await db.query(
              "SELECT name FROM roles WHERE id = ?",
              [resource.role]
            );
            return {
              role: resource.role,
              role_name: role?.[0]?.name,
              no_of_people: resource.no_of_people,
            };
          }
          return resource;
        })
      );

      return {
        ...row,
        crisis_response_actions: crisisResponseActions,
        immediate_decisions_made: immediateDecisionsMade,
        crisis_communications: {
          communications_sent: crisisCommunications.communications_sent,
          communication_details: communicationDetails,
        },
        resource_allocation: {
          human_resources: humanResources,
          technological_resources:
            resourceAllocation.technological_resources || {},
        },
      };
    })
  );

  return processedData;
};

export const getInvestigationDataIntoCompleted = async (id) => {
  const joins = [
    {
      targetTable: "department",
      onCondition: "investigation.department = department.id",
      type: "LEFT",
    },
    {
      targetTable: "organization",
      onCondition: "investigation.organization = organization.id",
      type: "LEFT",
    },
    {
      targetTable: "users as employeeUser",
      onCondition: "investigation.employee = employeeUser.id",
      type: "LEFT",
    },

    {
      targetTable: "users as created_user",
      onCondition: "investigation.created_by = created_user.id",
      type: "LEFT",
    },
    {
      type: "LEFT",
      targetTable: "department AS employee_user_department",
      onCondition: "employeeUser.department = employee_user_department.id",
    },
    {
      type: "LEFT",
      targetTable: "roles AS employee_user_role",
      onCondition: "employeeUser.role = employee_user_role.id",
    },
    {
      type: "left",
      targetTable: "incident",
      onCondition: "incident.id = investigation.incident_id",
    },
    {
      type: "left",
      targetTable: "severity",
      onCondition: "severity.id = investigation.actual_incident_severity",
    },
    // {
    //   type: "left",
    //   targetTable: "users as involved_person",
    //   onCondition: "involved_person.id = investigation.involved_person",
    // },
    {
      type: "left",
      targetTable: "repository as investigation_ddrm",
      onCondition:
        "investigation_ddrm.id = investigation.investigation_ddrm_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  const investigationGetQuery = `
    SELECT investigation.*, 
           incident.incident_title AS incident_title_name,
           employeeUser.name AS employee_name, 
           employeeUser.role AS employee_role,
           employeeUser.department AS employee_department,
           severity.name AS actual_incident_severity_name,
           employee_user_department.name AS employee_department_name,
           employee_user_role.name AS employee_role_name,
           created_user.name AS created_by_name, 
           created_user.surname AS created_by_surname, 
           organization.name AS organization_name,
           department.name AS department_name ,
           investigation_ddrm.url AS investigation_ddrm_file
    FROM investigation 
    ${joinsRecord}
    WHERE investigation.deleted = 0 AND investigation.id = ${id}`;

  let [getInvestigationRecord] = await db.query(investigationGetQuery);
  getInvestigationRecord = await decodeAndParseFields(getInvestigationRecord);
  // console.log("getInvestigationRecord: ", getInvestigationRecord);

  for (let i = 0; i < getInvestigationRecord.length; i++) {
    const investigation = getInvestigationRecord[i];

    if (investigation.incident_persons_involved && Array.isArray(investigation.incident_persons_involved)) {
      for (let j = 0; j < investigation.incident_persons_involved.length; j++) {
        const person = investigation.incident_persons_involved[j];

        if (person.person_type === "Internal") {
          if (person.user_id) {
            const [fetch] = await db.query(
              `SELECT CONCAT(users.name, ' ', users.surname) AS user_name, 
                      roles.name AS role_name, 
                      department.name AS department_name 
               FROM users 
               LEFT JOIN roles ON users.role = roles.id 
               LEFT JOIN department ON users.department = department.id 
               WHERE users.id = ${person.user_id}`
            );

            let incidentRole = [];
            if (person.internal_incident_role) {
              [incidentRole] = await db.query(
                `SELECT name as internal_incident_role_name 
                 FROM incident_role 
                 WHERE id = ${person.internal_incident_role}`
              );
            }

            investigation.incident_persons_involved[i] = {
              ...person,
              ...(fetch[0] || {}),
              ...(incidentRole[0] || {})
            };
          }
        }

        else if (person.person_type === "External") {
          if (person.is_contractor === '1' && person.contractor_id) {
            const [contractorData] = await db.query(
              `SELECT contractor_name as contractor_name 
               FROM contractor_registration 
               WHERE id = ${person.contractor_id}`
            );

            const [incidentRole] = await db.query(
              `SELECT name as external_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.external_incident_role}`
            );

            let projectData = {};
            if (person.project_id) {
              const [project] = await db.query(
                `SELECT id as project_id, project_title as project_name 
                 FROM project_registration 
                 WHERE id = ${person.project_id}`
              );
              projectData = project[0] || {};
            }

            investigation.incident_persons_involved[j] = {
              ...person,
              ...(contractorData[0] || {}),
              ...(incidentRole[0] || {}),
              ...projectData
            };
          } else {
            const [incidentRole] = await db.query(
              `SELECT name as external_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.external_incident_role}`
            );

            investigation.incident_persons_involved[j] = {
              ...person,
              ...(incidentRole[0] || {})
            };
          }
        }

        else if (person.person_type === "Both") {
          let userDetails = {};
          if (person.user_id) {
            const [fetch] = await db.query(
              `SELECT CONCAT(users.name, ' ', users.surname) AS user_name, 
                      roles.name AS role_name, 
                      department.name AS department_name 
               FROM users 
               LEFT JOIN roles ON users.role = roles.id 
               LEFT JOIN department ON users.department = department.id 
               WHERE users.id = ${person.user_id}`
            );
            userDetails = fetch[0] || {};
          }

          let internalRoleDetails = {};
          if (person.internal_incident_role) {
            const [internalRole] = await db.query(
              `SELECT name as internal_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.internal_incident_role}`
            );
            internalRoleDetails = internalRole[0] || {};
          }

          let externalDetails = {};
          let projectDetails = {};

          let externalRoleDetails = {};
          if (person.external_incident_role) {
            const [externalRole] = await db.query(
              `SELECT name as external_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.external_incident_role}`
            );
            externalRoleDetails = externalRole[0] || {};
          }

          if (person.is_contractor === '1' && person.contractor_id) {
            const [contractorData] = await db.query(
              `SELECT contractor_name as contractor_name 
               FROM contractor_registration 
               WHERE id = ${person.contractor_id}`
            );
            externalDetails = contractorData[0] || {};

            if (person.project_id) {
              const [project] = await db.query(
                `SELECT id as project_id, project_title as project_name 
                 FROM project_registration 
                 WHERE id = ${person.project_id}`
              );
              projectDetails = project[0] || {};
            }
          }

          investigation.incident_persons_involved[j] = {
            ...person,
            ...userDetails,
            ...internalRoleDetails,
            ...externalRoleDetails,
            ...externalDetails,
            ...projectDetails
          };
        }
      }
    }

    if (investigation.investigation_enquiry_details && Array.isArray(investigation.investigation_enquiry_details)) {
      for (let j = 0; j < investigation.investigation_enquiry_details.length; j++) {
        const person = investigation.investigation_enquiry_details[j];
        if (person.person_involved_internal_external === "0") {
          investigation.investigation_enquiry_details[j] = {
            ...person,
          };
        } else if (person.person_involved_internal_external === "1") {
          let userDetails = {};
          if (person.involved_person && typeof person.involved_person == "number") {
            const [fetch] = await db.query(
              `SELECT CONCAT(users.name, ' ', users.surname) AS involved_person, 
                      roles.name AS role_name, 
                      department.name AS department_name 
               FROM users 
               LEFT JOIN roles ON users.role = roles.id 
               LEFT JOIN department ON users.department = department.id 
               WHERE users.id = ${person.involved_person}`
            );
            userDetails = fetch[0] || {};
          }
          investigation.investigation_enquiry_details[j] = {
            ...person,
            ...userDetails,
          };
        }
      }
    }

    if (
      investigation?.evidence_collected &&
      investigation?.evidence_collected.length > 0 &&
      investigation?.evidence_collected[0]?.ddrm_id
    ) {
      for (let evidence of investigation?.evidence_collected) {
        const [repository] = await db.query(
          `SELECT repository.url, document_name as title FROM repository LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE repository.id = ?`,
          [evidence.ddrm_id]
        );
        evidence.file = repository[0]?.url;
        evidence.title = repository[0]?.title;

        try {
          evidence.base64File = await convertFileToBase64(repository[0].url);
        } catch (error) {
          console.error(`Failed to convert file to Base64: ${error.message}`);
        }
      }
    }
    if (
      investigation?.upload_document &&
      investigation?.upload_document.length > 0 &&
      investigation?.upload_document[0]?.ddrm_id
    ) {
      for (let uploadDoc of investigation?.upload_document) {
        const [repository] = await db.query(
          `SELECT repository.url, document_name as title FROM repository LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE repository.id = ?`,
          [uploadDoc.ddrm_id]
        );
        uploadDoc.file = repository[0]?.url;
        uploadDoc.title = repository[0]?.title;

        try {
          uploadDoc.base64File = await convertFileToBase64(repository[0].url);
        } catch (error) {
          console.error(`Failed to convert file to Base64: ${error.message}`);
        }
      }
    }

    if (investigation?.type_of_security_incident) {
      const [securityIncidentType] = await db.query(
        "SELECT name FROM security_incident_type WHERE id = ?",
        [investigation.type_of_security_incident]
      );
      investigation.type_of_security_incident_name =
        securityIncidentType[0]?.name;
    }

    if (
      investigation &&
      investigation?.any_arrests &&
      investigation?.any_arrests.length > 0
    ) {
      const [users] = await db.query(
        `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
          LEFT JOIN department ON department.id  = users.department WHERE users.id IN (?)`,
        [investigation.any_arrests]
      );
      investigation.any_arrests_details = users;
    }

    if (investigation?.security_currency) {
      const [securityCurrency] = await db.query(
        "SELECT name FROM currency WHERE id = ?",
        [investigation.security_currency]
      );
      investigation.security_currency_name = securityCurrency[0]?.name;
    }
    if (investigation?.estimated_currency) {
      const [securityCurrency] = await db.query(
        "SELECT name FROM currency WHERE id = ?",
        [investigation.estimated_currency]
      );
      investigation.estimated_currency_name = securityCurrency[0]?.name;
    }
    // for incident category name
    if (
      investigation?.incident_category &&
      investigation?.incident_category.length > 0
    ) {
      const [incidentCategory] = await db.query(
        "SELECT name FROM incident_category WHERE id IN (?)",
        [investigation.incident_category]
      );
      getInvestigationRecord[i].incident_category_name = incidentCategory;
    }

    /// for similar incidents
    if (
      investigation?.similar_incidents &&
      investigation?.similar_incidents.length > 0
    ) {
      const [similarIncidents] = await db.query(
        "SELECT id, incident_title as name FROM incident WHERE id IN (?)",
        [investigation.similar_incidents]
      );
      getInvestigationRecord[i].similar_incidents_name = similarIncidents;

      // for investigation submitted
      if (
        investigation?.investigation_submitted &&
        investigation?.investigation_submitted.length > 0
      ) {
        const investigation_submitted =
          investigation?.investigation_submitted.map((item) => item.employee);
        const investigationSubmitted = await getEmployeeByIds(
          investigation_submitted
        );
        getInvestigationRecord[i].investigation_submitted =
          investigationSubmitted;
      }
    }

    // for notify users
    if (investigation?.notify && investigation?.notify.length > 0) {
      const notify = investigation?.notify.map((item) => item.employee);
      const notifyUsers = await getEmployeeByIds(notify);
      getInvestigationRecord[i].notify = notifyUsers;
    }
    /// actual_incident_category get all related data

    if (
      investigation?.actual_incident_category &&
      investigation?.actual_incident_category.length > 0
    ) {
      const [actualIncidentCategory] = await db.query(
        "SELECT name FROM incident_category WHERE id IN (?)",
        [investigation.actual_incident_category]
      );
      getInvestigationRecord[i].actual_incident_category_name =
        actualIncidentCategory;
    }

    if (investigation.incident_id) {
      const { data } = await getIncidentInvestigationData(
        investigation.incident_id
      );
      getInvestigationRecord[i].incident_data = data?.records;
    }
    // injured_persons , vehicle_details, asset_type , quality_incident_details , incident_type , occupationals
    getInvestigationRecord[i].injured_persons = getInvestigationRecord[i]
      ?.injured_persons
      ? getInvestigationRecord[i].injured_persons
      : [];

    for (let injuredPerson of investigation.injured_persons) {
      if (injuredPerson && injuredPerson?.user_id) {
        const [user] = await db.query(
          `SELECT  CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [injuredPerson.user_id]
        );
        injuredPerson.user_name = user[0].name;
      }

      if (
        injuredPerson &&
        injuredPerson?.parts_of_body_affected &&
        injuredPerson?.parts_of_body_affected.length
      ) {
        const [bodyParts] = await db.query(
          `SELECT label, parent_id FROM body_part WHERE id IN (?)`,
          [injuredPerson.parts_of_body_affected]
        );
        injuredPerson.parts_of_body_affected_details = bodyParts;
      }

      if (injuredPerson && injuredPerson?.reported_by) {
        const [user] = await db.query(
          `SELECT  CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [injuredPerson.reported_by]
        );
        injuredPerson.reported_by_name = user[0].name;
      }
    }

    getInvestigationRecord[i].vehicle_details = getInvestigationRecord[i]
      .vehicle_details
      ? getInvestigationRecord[i].vehicle_details
      : [];

    for (let vehicleDetail of investigation.vehicle_details) {
      if (vehicleDetail && vehicleDetail?.reported_employee) {
        const [user] = await db.query(
          `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
          [vehicleDetail.reported_employee]
        );
        vehicleDetail.reported_employee_name = user[0].name;
        vehicleDetail.reported_employee_profile = user[0].profile;
        vehicleDetail.reported_employee_role = user[0].role;
        vehicleDetail.reported_employee_department = user[0].department;
      }
      if (vehicleDetail && vehicleDetail?.employee) {
        const [user] = await db.query(
          `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
          [vehicleDetail.employee]
        );
        vehicleDetail.employee_name = user[0].name;
        vehicleDetail.employee_profile = user[0].profile;
        vehicleDetail.employee_role = user[0].role;
        vehicleDetail.employee_department = user[0].department;
      }

      if (vehicleDetail && vehicleDetail?.currency) {
        const [currency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [vehicleDetail.currency]
        );
        vehicleDetail.currency_name = currency[0]?.name;
      }
      if (vehicleDetail && vehicleDetail?.actual_currency) {
        const [actualCurrency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [vehicleDetail.actual_currency]
        );
        vehicleDetail.actual_currency_name = actualCurrency[0]?.name;
      }
    }

    getInvestigationRecord[i].asset_type = getInvestigationRecord[i].asset_type
      ? getInvestigationRecord[i].asset_type
      : [];

    for (let assetType of investigation.asset_type) {
      if (assetType && assetType?.select_asset_type) {
        const [asset] = await db.query(
          `SELECT name FROM asset_type WHERE id = ?`,
          [assetType.select_asset_type]
        );
        assetType.select_asset_type_name = asset[0].name;
      }
      if (assetType && assetType?.select_asset) {
        const [asset] = await db.query(
          `SELECT asset_name FROM asset WHERE id = ?`,
          [assetType.select_asset]
        );
        assetType.select_asset_name = asset[0].asset_name;
      }

      if (assetType && assetType?.currency) {
        const [currency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [assetType.currency]
        );
        assetType.currency_name = currency[0]?.name;
      }
      if (assetType && assetType?.actual_currency) {
        const [actualCurrency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [assetType.actual_currency]
        );
        assetType.actual_currency_name = actualCurrency[0]?.name;
      }
      if (assetType && assetType?.damage_severity_level) {
        const [severity] = await db.query(
          `SELECT name FROM severity WHERE id = ?`,
          [assetType.damage_severity_level]
        );
        assetType.damage_severity_level_name = severity[0]?.name;
      }

      if (assetType && assetType?.reported_by_whom) {
        const [user] = await db.query(
          `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
          [assetType.reported_by_whom]
        );
        assetType.reported_by_whom_name = user[0].name;
        assetType.reported_by_whom_profile = user[0].profile;
        assetType.reported_by_whom_role = user[0].role;
        assetType.reported_by_whom_department = user[0].department;
      }
    }

    getInvestigationRecord[i].quality_incident_details = getInvestigationRecord[
      i
    ].quality_incident_details
      ? await getInvestigationRecord[i].quality_incident_details
      : [];

    getInvestigationRecord[i].incident_type = getInvestigationRecord[i]
      .incident_type
      ? getInvestigationRecord[i].incident_type
      : [];

    getInvestigationRecord[i].occupationals = getInvestigationRecord[i]
      .occupationals
      ? getInvestigationRecord[i].occupationals
      : [];

    for (let occupational of investigation.occupationals) {
      if (occupational && occupational?.exposed_person_name) {
        const [user] = await db.query(
          `SELECT id, CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [occupational.exposed_person_name]
        );
        occupational.exposed_person_id = user[0]?.id;
        occupational.exposed_person_name = user[0]?.name;
        occupational.exposed_person_profile = user[0]?.profile;
      }

      if (
        occupational &&
        occupational?.occupational_type &&
        occupational?.occupational_type.length
      ) {
        const [occupationalType] = await db.query(
          `SELECT name FROM illness_type WHERE id IN (?)`,
          [occupational.occupational_type]
        );
        occupational.occupational_type_details = occupationalType;
      }
      if (occupational && occupational.reported_by) {
        const [user] = await db.query(
          `SELECT CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [occupational.reported_by]
        );
        occupational.reported_by_name = user[0]?.name;
        occupational.reported_by_profile = user[0]?.profile;
      }
    }

    if (investigation?.investigation_standardise) {
      for (let standardise of investigation?.investigation_standardise) {
        if (standardise && standardise?.who) {
          const [user] = await db.query(
            `SELECT  CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
            [standardise.who]
          );
          standardise.who_name = user[0].name;
          standardise.who_profile = user[0].profile;
        }
      }
    }
  }

  return getInvestigationRecord[0];
};
export const setSuperAdminPermission = async () => {
  try {
    const permission = [];
    let [organizationList] = await db.query(
      `SELECT * FROM organization where deleted = 0`
    );
    let organization = organizationList.map((item) => item.id);
    let permissions = {};
    organization.forEach((item) => {
      permissions[item] = {
        view: true,
        edit: true,
        delete: true,
        create: true,
      };
    });

    // get all sidebar module list which are not deleted
    const [record] = await db.query(`SELECT * FROM sidebar WHERE deleted = 0`);
    const sidebarList = record.map((item) => item.id);

    sidebarList.forEach((item) => {
      permission.push({
        sidebarId: item,
        permissions: permissions,
      });
    });
    const [result] = await db.query(
      `UPDATE users SET permission='${JSON.stringify(
        permission
      )}', organization = '${JSON.stringify(organization)}' WHERE id='1' `
    );
    console.log("permission set done");
  } catch (error) {
    console.log("Error While Setting Up super admin permission");
  }
};

export function incrementVersion(version) {
  let [major, minor, patch] = version?.split(".").map(Number);
  patch += 1;
  // If patch reaches 10, reset it to 0 and increment the minor version
  if (patch === 10) {
    patch = 0;
    minor += 1;
  }
  return `${major}.${minor}.${patch}`;
}

export const createStaticDataSettings = async (
  model,
  organizationId,
  columns = ["name", "description"]
) => {
  // const keys = columns.join(", ");
  // these data are static we should filter on the basis of any organization
  // const [organizationList] = await db.query(
  //   `SELECT * FROM organization Where deleted = 0`
  // );
  // const organization = organizationList[0].id;
  try {
    // if data type is JSON
    // const [result] = await db.query(
    //   `UPDATE ${model.tableName} SET organization = JSON_MERGE(organization, '[${organizationId}]') WHERE NOT JSON_CONTAINS(ids, '${organizationId}') AND deleted = 0`
    // );
    const [result] = await db.query(
      `UPDATE ${model.tableName} SET organization = CONCAT(LEFT(organization, LENGTH(organization) - 1),',${organizationId}]') WHERE organization LIKE '[%]' AND deleted = 0`
    );
    // const [result] = await db.query(
    //   `SELECT organization FROM ${model.tableName} where deleted = 0`
    // );
    // if (result.length !== 0) {
    //   for (let item of result) {
    //     const { query, values } = createQueryBuilder(model, {
    //       ...item,
    //       organization: organizationId,
    //     });
    //     await db.query(query, values);
    //   }

    // Manually using nodejs not query

    //   const [result] = await db.query(
    //     `SELECT id,organization FROM ${model.tableName} WHERE deleted = 0`
    //   )

    // for (let row of result) {
    //   let currentIds = JSON.parse(row.organization);

    //   if (!currentIds.includes(organizationId)) {
    //     currentIds.push(organizationId);
    //   }

    //   await db.query(`UPDATE ${model.tableName} SET organization = ? WHERE id = ?`, [currentIds, row.id]);
    // }

    console.log("Static data settings created successfully");
    // }
  } catch (error) {
    console.log("Error while createStaticDataSettings :", error);
  }
};
// await createStaticDataSettings(IncidentRole, 36);

export const createStaticDataForModels = async (
  models,
  organizationId,
  columns = ["name", "description"]
) => {
  const keys = columns.join(", ");

  // Fetch organizations where deleted = 0
  const [organizationList] = await db.query(
    `SELECT * FROM organization WHERE deleted = 0`
  );

  const organization = organizationList[0]?.id;
  try {
    for (let model of models) {
      const [result] = await db.query(
        `SELECT ${keys} FROM ${model.tableName} WHERE deleted = 0 AND AND organization = ${organization} AND is_static = 1`
      );

      if (result.length !== 0) {
        for (let item of result) {
          const { query, values } = createQueryBuilder(model, {
            ...item,
            organization: organizationId,
          });
          await db.query(query, values);
        }
        console.log(`Static data inserted for model: ${model.tableName}`);
      }
    }
    console.log("Static data settings created successfully for all models.");
  } catch (error) {
    console.log("Error while creating static data settings:", error);
  }
};

export const convertFileToBase64 = (filePath) => {
  const publicDir = path.join(process.cwd(), "public");
  filePath = path.join(publicDir, filePath);
  return new Promise((resolve, reject) => {
    fs.readFile(filePath, (err, data) => {
      if (err) {
        reject(new Error(`Failed to read file at ${filePath}: ${err.message}`));
        return;
      }

      const ext = path.extname(filePath).split(".")[1];
      // Convert file buffer to Base64
      const base64 = data.toString("base64");
      // Construct the Base64 string with the data URI prefix
      const base64WithMime = `data:image/${ext};base64,${base64}`;
      resolve(base64WithMime);
    });
  });
};

const softDeleteRelatedRecords = async (
  model = ContractorRegistration,
  id = 7
) => {
  try {
    const whereClause = { id };
    // await model.update({ deleted: 1 }, { where: { id } });

    const models = sequelize.models;
    console.log("models: ", models);

    // const relatedModels = Object.values(sequelize.models).filter((relatedModel) =>
    //   Object.values(relatedModel.associations).some((association) => association.target === Role)
    // );
    // console.log("relatedModels: ", relatedModels);

    // for (const relatedModel of relatedModels) {
    //   const relatedAssociations = Object.values(relatedModel.associations).filter(
    //     (association) => association.target === model
    //   );

    // for (const association of relatedAssociations) {
    //   const foreignKey = association.foreignKey;
    //   const condition = {
    //     [foreignKey]: {
    //       [Op.in]: await model
    //         .findAll({
    //           attributes: ["id"],
    //           where: whereClause,
    //           raw: true,
    //         })
    //         .then((records) => records.map((record) => record.id)),
    //     },
    //   };
    //   console.log("condition: ", condition);
    // }
    // }
  } catch (error) {
    throw error;
  }
};

// softDeleteRelatedRecords();

export const getDateRangeForSQL = (
  dateRange,
  customStartDate = null,
  customEndDate = null
) => {
  const today = moment().endOf("day");
  let startDate, endDate;

  switch (dateRange) {
    case "All-Time":
      // Set to a fixed past date for "All-Time"
      startDate = moment("1900-01-01").startOf("day");
      endDate = today;
      break;

    case "Year To Date":
      startDate = moment().startOf("year");
      endDate = today;
      break;

    case "Last 12 Months":
      startDate = moment().subtract(12, "months").startOf("day");
      endDate = today;
      break;

    case "Previous Year":
      startDate = moment().subtract(1, "year").startOf("year");
      endDate = moment().subtract(1, "year").endOf("year");
      break;

    case "Custom Date":
      if (customStartDate && customEndDate) {
        startDate = moment(customStartDate).startOf("day");
        endDate = moment(customEndDate).endOf("day");
      } else {
        throw new Error("Custom date range requires both start and end dates");
      }
      break;

    default:
      startDate = today.clone().startOf("day");
      endDate = today;
      break;
  }

  // Format dates for SQL
  const formattedDates = {
    startDate: startDate.format("YYYY-MM-DD HH:mm:ss"),
    endDate: endDate.format("YYYY-MM-DD HH:mm:ss"),
  };

  // Helper function to generate SQL WHERE clause
  const getSQLWhereDateClause = (dateColumnName = "created_at") => {
    return `${dateColumnName} BETWEEN '${formattedDates.startDate}' AND '${formattedDates.endDate}'`;
  };

  return {
    ...formattedDates,
    getSQLWhereDateClause,
  };
};

export const uploadToS3 = async (file) => {
  // AWS.config.update({
  //   region: process.env.AWS_REGION,
  //   accessKeyId: process.env.AWS_ACCESS_KEY_ID,
  //   secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
  // });
  const s3 = new AWS.S3({
    accessKeyId: process.env.AWS_ACCESS_KEY_ID,
    secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY,
    region: process.env.AWS_REGION,
  });

  const bucketName = process.env.AWS_S3_BUCKET;

  const params = {
    Bucket: bucketName,
    Key: `${Date.now()}-${file.name}`,
    Body: file.data,
    ContentType: file.mimetype,
    ACL: "public-read",
  };

  try {
    const uploadResult = await s3.upload(params).promise();
    console.log("AWS S3 File Upload Success:", uploadResult.Location);
    return uploadResult.Location;
  } catch (error) {
    console.error("AWS S3 Upload Error:", error);
    throw error;
  }
};

export const uploadToAzure = async (filePath, fileName) => {
  const blobServiceClient = BlobServiceClient.fromConnectionString(
    process.env.AZURE_STORAGE_CONNECTION_STRING
  );
  const containerClient = blobServiceClient.getContainerClient(
    process.env.AZURE_CONTAINER_NAME
  );

  const blockBlobClient = containerClient.getBlockBlobClient(fileName);

  try {
    const uploadBlobResponse = await blockBlobClient.uploadFile(filePath);
    console.log("Azure Blob Upload Success:", uploadBlobResponse);
    return blockBlobClient.url;
  } catch (error) {
    console.error("Azure Blob Upload Error:", error);
    throw error;
  }
};
export const uploadToGCP = async (filePath, fileName) => {
  const publicDir = path.join(process.cwd(), "public");
  const storage = new Storage({
    projectId: process.env.GCP_PROJECT_ID,
    keyFilename: `${publicDir}${process.env.GCP_SERVICE_ACCOUNT_KEY}`,
  });

  const bucket = storage.bucket(process.env.GCP_STORAGE_BUCKET);

  try {
    const uploadOptions = {
      destination: fileName,
      contentType: "application/octet-stream",
      // Remove the public: true option
      metadata: {
        cacheControl: 'public, max-age=31536000',
      },
    };
    const uploadResult = await bucket.upload(filePath, uploadOptions);
    console.log(
      "Google Cloud Upload Success:",
      uploadResult[0].metadata.mediaLink
    );
    const [signedUrl] = await uploadResult[0].getSignedUrl({
      action: 'read',
      expires: Date.now() + (1000 * 60 * 60 * 24 * 365 * 100), // 100 years
    });
    return signedUrl;
  } catch (error) {
    console.error("Google Cloud Upload Error:", error);
    throw error;
  }
};

export const EntryInSuperAdmin = async (sessionId) => {
  try {
    const [data] = await getRecord("users", "id", sessionId);
    const { email, first_name, surname } = data;
    // get the super admin client internal id for for identity purpose
    const adminQuery = `SELECT client_internal_id FROM user where client_internal_id IS NOT NULL `;
    const [adminRecord] = await db.query(adminQuery);
    const client_internal_id = adminRecord[0]?.client_internal_id;
    const bodyData = {
      email,
      username: `${first_name} ${surname}`,
      clientInternalId: client_internal_id,
    };

    const response = await axios.post(
      `${process.env.AUTH0_DOMAIN}/api/unified-user/`,
      bodyData
    );
    console.log("Success", response.data);
  } catch (error) {
    console.log("Error From : EntryInSuperAdmin", error);
  }
};

export const getFirstCreatedAndAllOrganizationIds = async () => {
  try {
    const query = `
    SELECT
        (SELECT id FROM organization WHERE deleted = 0 ORDER BY created_at ASC LIMIT 1) AS firstOrgId,
        GROUP_CONCAT(id) AS organizationIds
    FROM organization
    WHERE deleted = 0`;
    const [[result]] = await db.query(query);
    // Convert `organizationIds` from comma-separated string to an array
    if (result.organizationIds) {
      result.organizationIds = result.organizationIds.split(",").map(Number);
    } else {
      result.organizationIds = [];
    }
    return result;
  } catch (error) {
    throw error;
  }
};

export const getFilterConditions = async (req, db, table) => {
  let organizationFilter;
  let organizationCondition = "";
  let condition = "";

  try {
    const filter = JSON.parse(req.query?.filter || "{}");

    if (filter.department) {
      const [orgDetails] = await db.query(
        `SELECT id, organization FROM department WHERE id IN (?)`,
        [filter.department]
      );
      organizationFilter = orgDetails?.[0]?.organization || null;
    } else {
      organizationFilter = filter.organization || null;
    }

    if (organizationFilter) {
      const orgFilter = Array.isArray(organizationFilter)
        ? JSON.stringify(organizationFilter)
        : `${organizationFilter}`;
      organizationCondition = ` AND JSON_CONTAINS(${table}.organization, '${orgFilter}') `;
    }

    // Generate additional WHERE conditions dynamically
    condition = await whereCondition({
      table,
      page: req.query.page,
      all: req.query.all,
      pageSize: req.query.pageSize,
      id: req.params.id,
      // grouped: req.query.grouped,
      user: req.user,
    });
  } catch (error) {
    console.error("Error parsing filter:", error);
  }

  return { organizationCondition, condition };
};

export const getListingData = async (elements) => {
  for (let element of elements) {
    element.description = await decodeSingle_statement(element.description);
    if (element.organizations) {
      let orgIds = [];

      try {
        const parsedOrgs = JSON.parse(element.organizations);
        if (Array.isArray(parsedOrgs)) {
          orgIds = [...new Set(parsedOrgs)];
        }
      } catch (error) {
        console.error(
          "Error parsing organization:",
          element.organizations,
          error
        );
      }

      if (orgIds.length > 0) {
        const [orgDetails] = await db.query(
          `SELECT id, name, business_logo as image FROM organization WHERE id IN (?)`,
          [orgIds]
        );

        const orgDetailsMap = new Map(orgDetails.map((org) => [org.id, org]));
        element.organization_details = orgIds.map(
          (orgId) => orgDetailsMap.get(orgId) || { id: orgId, image: "" }
        );
      }

      // Remove the 'organizations' key
      delete element.organizations;
    }
  }
  elements = await decodeAndParseFields(elements);
  return elements;
};

/**
 * Modifies a table's organization column from a single ID to LONGTEXT format with JSON array
 * and updates all existing records to use the new format
 *
 * @param {string} tableName - The name of the table to modify
 * @returns {Promise<boolean>} - Returns true if successful, throws error otherwise
 */
export async function updateOrganizationColumnFormat(tableName) {
  try {
    // Start a transaction to ensure all operations succeed or fail together
    await db.query("START TRANSACTION");

    // Step 1: Get the foreign key constraints on the organization column
    const [foreignKeys] = await db.query(
      `
      SELECT CONSTRAINT_NAME 
      FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
      WHERE TABLE_NAME = ? 
      AND COLUMN_NAME = 'organization' 
      AND CONSTRAINT_NAME <> 'PRIMARY' 
      AND REFERENCED_TABLE_NAME IS NOT NULL
    `,
      [tableName]
    );

    // Step 2: Drop foreign key constraints if they exist
    for (const fk of foreignKeys) {
      try {
        await db.query(
          `ALTER TABLE \`${tableName}\` DROP FOREIGN KEY \`${fk.CONSTRAINT_NAME}\``
        );
        console.log(`Dropped foreign key: ${fk.CONSTRAINT_NAME}`);
      } catch (fkError) {
        // If the foreign key doesn't exist, just log and continue
        if (fkError.code === "ER_CANT_DROP_FIELD_OR_KEY") {
          console.log(
            `Foreign key ${fk.CONSTRAINT_NAME} doesn't exist, skipping`
          );
        } else {
          throw fkError;
        }
      }
    }

    // Step 3: Get all constraint names from INFORMATION_SCHEMA to check indexes
    const [allConstraints] = await db.query(`
      SHOW INDEX FROM \`${tableName}\`
      WHERE Column_name = 'organization'
    `);

    // Drop indexes on organization column
    for (const idx of allConstraints) {
      if (idx.Key_name !== "PRIMARY") {
        try {
          await db.query(`DROP INDEX \`${idx.Key_name}\` ON \`${tableName}\``);
          console.log(`Dropped index: ${idx.Key_name}`);
        } catch (idxError) {
          if (idxError.code === "ER_CANT_DROP_FIELD_OR_KEY") {
            console.log(`Index ${idx.Key_name} doesn't exist, skipping`);
          } else {
            throw idxError;
          }
        }
      }
    }

    // Step 4: Change column type to LONGTEXT
    await db.query(
      `ALTER TABLE \`${tableName}\` CHANGE COLUMN \`organization\` \`organization\` LONGTEXT NULL`
    );
    console.log(
      `Changed organization column type to LONGTEXT in table: ${tableName}`
    );

    // Step 5: Update existing records to use JSON array format
    await db.query(`
      UPDATE \`${tableName}\` 
      SET organization = CONCAT('[', organization, ']') 
      WHERE organization IS NOT NULL 
      AND TRIM(organization) <> ''
      AND organization NOT LIKE '[%'
    `);
    console.log(`Updated existing records in table: ${tableName}`);

    // Commit the transaction
    await db.query("COMMIT");
    console.log(
      `Successfully updated organization column format for table: ${tableName}`
    );

    return true;
  } catch (error) {
    // Rollback in case of error
    await db.query("ROLLBACK");
    console.error(
      `Error updating organization column for table: ${tableName}`,
      error
    );
    throw error;
  }
}

export const backupDatabase = async () => {
  try {
    const { DATABASE_HOST, DATABASE_USER, DATABASE_PASSWORD, DATABASE_NAME } =
      process.env;

    if (!DATABASE_HOST || !DATABASE_USER || !DATABASE_NAME) {
      throw new Error(
        "Missing required database credentials in environment variables."
      );
    }

    const cwd = process.cwd();
    const backupFolder = path.join(cwd, "backUpDatabase");
    if (!fs.existsSync(backupFolder)) {
      fs.mkdirSync(backupFolder, { recursive: true });
    }
    const date = new Date();
    const formattedDate = `${date.getFullYear()}-${(date.getMonth() + 1)
      .toString()
      .padStart(2, "0")}-${date.getDate().toString().padStart(2, "0")}`;
    const backupFileName = `${DATABASE_NAME}_${formattedDate}.sql`;
    const backupFilePath = path.join(backupFolder, backupFileName);

    // Create SQL dump
    await mysqldump({
      connection: {
        host: DATABASE_HOST,
        user: DATABASE_USER,
        password: DATABASE_PASSWORD || "",
        database: DATABASE_NAME,
      },
      dumpToFile: backupFilePath,
    });

    console.log(
      `Database backup created successfully at==`,
      `${backupFilePath}`
    );
  } catch (error) {
    console.error(`Backup error==`, `${error.message}`);
  }
};

export async function updateTeamMembers(teamId, userId) {
  try {
    const [team] = await db.query(
      `SELECT id, team_members FROM teams where id = ${teamId} AND deleted = 0`
    );
    if (team.length == 0) {
      // sendResponse(res, 404, "Team does not exist")
      console.log("Team does not exist");
      return;
    }
    const members = team[0].team_members ? team[0].team_members : [];

    if (members && members.includes(userId)) {
      // sendResponse(res, 400, "Member already exist in this team")
      console.log("Member already exist in this team.");
      return;
    }
    const [existUsers] = await db.query(
      `SELECT id, team_members
       FROM teams
       WHERE deleted = 0
       AND JSON_CONTAINS(team_members, CAST(? AS JSON))`,
      [userId]
    );

    if (existUsers.length) {
      for (let team of existUsers) {
        const updatedTeamMembers = team.team_members.filter(
          (member) => member !== userId
        );
        const stringifyMembers = JSON.stringify(updatedTeamMembers);
        await db.query(
          `UPDATE teams SET team_members = '${stringifyMembers}' WHERE id = ${team.id}`
        );
      }
    }
    members.push(userId);
    const stringifyMembers = JSON.stringify(members);
    await db.query(
      `UPDATE teams SET team_members = '${stringifyMembers}' WHERE id = ${teamId}`
    );
  } catch (error) {
    console.log("Error in appending team members: ", error);
    throw error;
  }
}

export const getUserById = async (id) => {
  const query = `SELECT users.id, CONCAT(users.name , ' ' , users.surname) as name, users.email, organization.name as company_name, organization.business_logo as company_logo FROM users LEFT JOIN organization ON organization.id = users.my_organization WHERE users.id = ${id} AND users.deleted="0"`;
  const [result] = await db.query(query);
  return result[0];
};

export async function getManagerAndSuperAdmin(managerId = null) {
  const usersToSend = [];
  if (managerId) {
    const manager = await getUserById(managerId);
    usersToSend.push(manager);
  }
  const superAdmin = await getUserById(1);
  usersToSend.push(superAdmin);
  return usersToSend;
}

export async function getCompanyName(orgId) {
  const [result] = await db.query(
    `SELECT organization.name, organization.business_logo FROM organization WHERE organization.id = ? AND deleted = "0"`,
    [orgId]
  );
  return result[0] ?? "";
}

export async function getModuleAndSubModule(moduleId, subModuleId) {
  const [result] = await db.query(
    `
    SELECT 
    m.title AS module_name, 
    sm.title AS submodule_name
    FROM sidebar m
    LEFT JOIN sidebar sm ON sm.id = ? AND sm.deleted = "0"
    WHERE m.id = ? AND m.deleted = "0"
    `,
    [subModuleId, moduleId]
  );
  return result[0];
}

// sendTerminateNotification()
export async function sendTerminateNotification() {
  try {
    const today = moment().format("YYYY-MM-DD");

    const termDate = moment(today).format("DD-MM-YYYY");
    const companyWebsite = process.env.HARMONY_FRONTEND_URL;

    const [users] = await db.query(
      `
      SELECT 
          users.id,
          users.unique_id AS employee_unique_id,
          users.my_organization,
          users.manager as manager_id,
          CONCAT(users.name, ' ', users.surname) AS employee_name,
          users.date_of_termination,
          department.name AS department,
          employee_type.name AS employee_type,
          teams.name as team_name,
          GROUP_CONCAT(
              CONCAT(
                  '{"action_id": ', IFNULL(custom_action_creation.id, 'null'), 
                  ', "action_name": "', IFNULL(custom_action_creation.task_title, ''), '"',
                  ', "action_status": "', IFNULL(custom_action_creation.status, ''), '"',
                  ', "action_end_date": "', IFNULL(custom_action_creation.end_date, ''), '"',
                  ', "module_id": ', IFNULL(custom_action_creation.module, 'null'),
                  ', "sub_module_id": ', IFNULL(custom_action_creation.sub_module, 'null'), 
                  '}'
              ) SEPARATOR ','
          ) AS actions
      FROM users
      LEFT JOIN department ON department.id = users.department
      LEFT JOIN employee_type ON employee_type.id = users.employee_type
      LEFT JOIN teams ON teams.id = users.team_id
      LEFT JOIN custom_action_creation ON custom_action_creation.responsible_person = users.id
      WHERE users.deleted = "0"
      AND users.date_of_termination IS NOT NULL
      AND DATE(users.date_of_termination) = ?
      GROUP BY users.id;
  `,
      [today]
    );

    users.forEach((user) => {
      user.actions = user.actions ? JSON.parse(`[${user.actions}]`) : [];
    });

    for (let terminatedUser of users) {
      const {
        id,
        employee_unique_id,
        employee_name,
        department,
        employee_type,
        team_name,
        my_organization,
      } = terminatedUser;
      const updateUser = await db.query(`UPDATE users SET employment_status = "Inactive" WHERE id = ${id}`)
      const departmentOrTeam = department ? department : team_name;
      const usersToSend = await getManagerAndSuperAdmin(
        terminatedUser.manager_id
      );
      const companyName = await getCompanyName(my_organization);

      for (let user of usersToSend) {
        const terminationEmailData = {
          templateFileUrl: "mail_for_employee_termination_template.html",
          employee_name: employee_name,
          name: user.name,
          employee_type: employee_type,
          emplyee_id: employee_unique_id,
          department_or_team: departmentOrTeam,
          company_website: companyWebsite,
          company_name: companyName,
          termination_date: termDate,
          templateName: "Employee Termination",
        };

        await sendEmail(
          "info@harmonyandhelp.com",
          user.email,
          // "mohammad.tariq.sartia@gmail.com",
          `Employee Termination Notice - ${employee_name}`,
          terminationEmailData
        );

        if (
          terminatedUser.actions &&
          terminatedUser.actions.length &&
          terminatedUser.actions[0].action_id
        ) {
          // console.log("terminatedUser.actions: ", terminatedUser.actions);
          let tableRows = await Promise.all(
            terminatedUser.actions.map(async (action) => {
              const result = await getModuleAndSubModule(
                action.module_id,
                action.sub_module_id
              );
              const { module_name, submodule_name } = result;

              return `
                    <tr>
                        <td>${action.action_name}</td>
                        <td>${module_name}</td>
                        <td>${submodule_name}</td>
                        <td>${moment(action.action_end_date).format(
                "DD-MM-YYYY"
              )}</td>
                        <td>${action.action_status}</td>
                    </tr>
                    `;
            })
          );

          const tableHtml = `
            <div class="table-container">
                <table>
                        <tr>
                            <th>Action Name</th>
                            <th>Module</th>
                            <th>Submodule</th>
                            <th>Due Date</th>
                            <th>Status</th>
                        </tr>
                        ${tableRows.join("")}
                </table>
            </div>
            `;

          const outstandingTasksEmailData = {
            templateFileUrl:
              "mail_for_employee_outstanding_tasks_template.html",
            employee_name: employee_name,
            name: user.name,
            termination_date: termDate,
            company_website: companyWebsite,
            company_name: companyName,
            modules: tableHtml,
            templateName: "Employee Outstanding Tasks",
          };

          await sendEmail(
            "info@harmonyandhelp.com",
            user.email,
            // "mohammad.tariq.sartia@gmail.com",
            `Outstanding Tasks for ${employee_name}`,
            outstandingTasksEmailData
          );
        }
      }
    }
  } catch (error) {
    throw error;
  }
}

export async function getPlatform() {
  try {
    const fullUrl = `${superadminApiUrl}/api/platform`;
    const { data } = await axios.get(fullUrl);
    console.log("data ----", data.data[0].detail);
    return data.data[0].detail;
  } catch (error) {
    console.error("Error sending payload:", error?.response || error.message);
    return;
  }
}

/**
 * Helper function to send emails dynamically.
 *
 * @param {Object} options - Email options.
 * @param {string} options.to - Recipient email.
 * @param {string} options.subject - Email subject.
 * @param {Object} options.data - Dynamic data for the template.
 * @param {string} options.data.templateFileUrl - Path to email template.
 */
export async function sendDynamicEmail({ to, subject, data }) {
  try {
    const info = await sendEmail("info@harmonyandhelp.com", to, subject, data);
    return info;
  } catch (error) {
    console.error("[!] Email sending failed:", error.message);
    throw error;
  }
}

export function generateStrategicTacticalTables(data) {
  let riskIdentificationDetails = data.risk_identification_details || [];

  const LIKELIHOOD_RATING = [
    { title: "E (100%)", value: 100 },
    { title: "E (90%)", value: 90 },
    { title: "D (80%)", value: 80 },
    { title: "D (70%)", value: 70 },
    { title: "D (60%)", value: 60 },
    { title: "D (50%)", value: 50 },
    { title: "C (40%)", value: 40 },
    { title: "C (30%)", value: 30 },
    { title: "C (20%)", value: 20 },
    { title: "B (10%)", value: 10 },
    { title: "A (1%)", value: 1 },
  ];

  //  const CONTROL_EFFECTIVENESS = [
  //   { title: 'Satisfactory (99%)', value: 99 },
  //   { title: 'Satisfactory (90%)', value: 90 },
  //   { title: 'Satisfactory (80%)', value: 80 },
  //   {
  //     title: 'Requires Improvement (79%)',
  //     value: 79
  //   },
  //   {
  //     title: 'Requires Improvement (70%)',
  //     value: 70
  //   },
  //   {
  //     title: 'Requires Improvement (60%)',
  //     value: 60
  //   },
  //   {
  //     title: 'Requires Improvement (50%)',
  //     value: 50
  //   },
  //   { title: 'Unsatisfactory (49%)', value: 49 },
  //   { title: 'Unsatisfactory (40%)', value: 40 },
  //   { title: 'Unsatisfactory (30%)', value: 30 },
  //   { title: 'Unsatisfactory (20%)', value: 20 },
  //   { title: 'Unsatisfactory (10%)', value: 10 },

  //   { title: 'None (9%)', value: 9 }
  // ];

  let strategicRiskIdentifiedRows = riskIdentificationDetails
    .map(
      (risk) => `
    <tr>
        <td>${risk.risk_name}</td>
        <td>${risk.risk_description}</td>
        <td>${risk.risk_owner_name} ${risk.risk_owner_surname}</td>
    </tr>
  `
    )
    .join("");

  const strategicRiskIdentifiedTable = `
    <div class="table-container">
        <table>
            <tr>
                <th>Risk Name</th>
                <th>Risk Description</th>
                <th>Risk Owner</th>
            </tr>
            ${strategicRiskIdentifiedRows}
        </table>
    </div>
  `;

  let inherentRiskRows = riskIdentificationDetails
    .map((risk) => {
      let likelihoodTitle =
        LIKELIHOOD_RATING.find(
          (item) => item.value === risk.inherent_likelihood
        )?.title || risk.inherent_likelihood;

      return `
    <tr>
        <td>${likelihoodTitle}</td>
        <td>${risk.inherent_consequence}</td>
        <td>${risk.inherent_risk_rating}</td>
        <td>${risk.inherent_risk_ranking}</td>
    </tr>
  `;
    })
    .join("");

  const inherentRiskTable = `
    <div class="table-container">
        <table>
            <tr>
                <th>Likelihood</th>
                <th>Consequence</th>
                <th>Risk Rating</th>
                <th>Risk Ranking</th>
            </tr>
            ${inherentRiskRows}
        </table>
    </div>
  `;

  let overallResidualRatingRows = riskIdentificationDetails
    .map(
      (risk) => `
      <tr>
          <td>${risk.residual_risk_rating}</td>
          <td>${risk.residual_risk_ranking}</td>
      </tr>
    `
    )
    .join("");

  const overallResidualRatingTable = `
    <div class="table-container">
        <table>
            <tr>
                <th>Risk Rating</th>
                <th>Risk Ranking</th>
            </tr>
            ${overallResidualRatingRows}
        </table>
    </div>
  `;

  // Control Identification Table (Handles multiple risks & multiple controls per risk)
  let controlIdentificationRows = riskIdentificationDetails
    .map((risk) =>
      (risk.control_identification || risk.tactical_identification || [])
        .map(
          (control) => `
      <tr>
          <td>${control.control_name}</td>
          <td>${control.control_design_intent}</td>
      </tr>
    `
        )
        .join("")
    )
    .join("");

  const controlIdentificationTable = controlIdentificationRows
    ? `
    <div class="table-container">
        <table>
            <tr>
                <th>Control Name</th>
                <th>Control Design Intent</th>
            </tr>
            ${controlIdentificationRows}
        </table>
    </div>
  `
    : "";

  // Residual Rating Table (Handles multiple risks & multiple controls per risk)
  let residualRatingRows = riskIdentificationDetails
    .map((risk) =>
      (risk.control_identification || risk.tactical_identification || [])
        .map(
          (control) => `
      <tr>
          <td>${control.residual_risk_rating}</td>
          <td>${control.residual_risk_ranking}</td>
      </tr>
    `
        )
        .join("")
    )
    .join("");

  const residualRatingTable = residualRatingRows
    ? `
    <div class="table-container">
        <table>
            <tr>
                <th>Risk Rating</th>
                <th>Risk Ranking</th>
            </tr>
            ${residualRatingRows}
        </table>
    </div>
  `
    : "";

  // Opportunity Identification Table (Handles multiple risks & multiple opportunities per risk)
  let opportunityIdentificationRows = riskIdentificationDetails
    .map((risk) =>
      (risk.opportunity_description || [])
        .map(
          (opportunity) => `
      <tr>
          <td>${opportunity.opportunity_name}</td>
          <td>${opportunity.description}</td>
      </tr>
    `
        )
        .join("")
    )
    .join("");

  const opportunityIdentificationTable = opportunityIdentificationRows
    ? `
    <div class="table-container">
        <table>
            <tr>
                <th>Opportunity Name</th>
                <th>Opportunity Description</th>
            </tr>
            ${opportunityIdentificationRows}
        </table>
    </div>
  `
    : "";

  return {
    strategicRiskIdentifiedTable,
    inherentRiskTable,
    controlIdentificationTable,
    residualRatingTable,
    opportunityIdentificationTable,
    overallResidualRatingTable,
  };
}

export async function getStrategicRisk(date, id) {
  console.log("date: ", date);
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = strategic_risk.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = strategic_risk.organization",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = strategic_risk.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as facilitator",
      onCondition:
        "facilitator.id = strategic_risk.risk_assessment_facilitator",
    },
    {
      type: "left",
      targetTable: "users as approver",
      onCondition: "approver.id = strategic_risk.risk_register_approver",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  let strategicRiskQuery = `SELECT
  strategic_risk.reject_comment, 
  strategic_risk.approval_status,
  strategic_risk.id,
  strategic_risk.unique_id,
   strategic_risk.reference_no,
   strategic_risk.created_by as created_by_id,
   strategic_risk.organization,
   organization.name as organization_name,
   users.name as created_by,
   users.surname as created_by_surname , users.profile as created_by_profile,strategic_risk.organization,strategic_risk.risk_register_name, strategic_risk.assessment_date , strategic_risk.review_date , strategic_risk.risk_context , strategic_risk.risk_assessment_facilitator , strategic_risk.register_assessment_facilitator , strategic_risk.risk_register_approver , strategic_risk.meeting_id ,strategic_risk.status, meeting.meeting_title as meeting , facilitator.name as facilitator_name , facilitator.surname as facilitator_surname , facilitator.profile as facilitator_profile ,facilitator.email as facilitator_email, approver.name as approver_name,approver.surname as approver_surname ,approver.profile as approver_profile, approver.email as approver_email 
      FROM strategic_risk 
      ${joinsRecord} where strategic_risk.deleted = 0`;
  if (date) {
    strategicRiskQuery += ` AND DATE(strategic_risk.review_date) = '${date}'`;
  }
  if (id) {
    strategicRiskQuery += ` AND strategic_risk.id = '${id}'`;
  }
  let [strategicRisk] = await db.query(strategicRiskQuery);

  strategicRisk = await decodeAndParseFields(strategicRisk);
  return strategicRisk;
}

export async function getTacticalRisk(date, id) {
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = tactical_risk.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = tactical_risk.organization",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = tactical_risk.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as facilitator",
      onCondition: "facilitator.id = tactical_risk.risk_assessment_facilitator",
    },
    {
      type: "left",
      targetTable: "users as approver",
      onCondition: "approver.id = tactical_risk.risk_register_approver",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  let tacticalRiskQuery = `SELECT
   tactical_risk.unique_id,
   tactical_risk.reference_no, 
  tactical_risk.reject_comment,tactical_risk.approval_status ,  tactical_risk.id, tactical_risk.created_by as created_by_id, tactical_risk.organization, organization.name as organization_name, users.name as created_by, users.surname as created_by_surname, users.profile as created_by_profile, tactical_risk.risk_register_name, tactical_risk.assessment_date, tactical_risk.review_date, tactical_risk.risk_context, tactical_risk.risk_assessment_facilitator, tactical_risk.register_assessment_facilitator, tactical_risk.risk_register_approver, tactical_risk.meeting_id, tactical_risk.status, meeting.meeting_title as meeting, facilitator.name as facilitator_name, facilitator.surname as facilitator_surname, facilitator.profile as facilitator_profile,facilitator.email as facilitator_email, approver.name as approver_name, approver.surname as approver_surname, approver.profile as approver_profile, approver.email as approver_email
      FROM tactical_risk 
      ${joinsRecord} WHERE tactical_risk.deleted = 0`;

  if (date) {
    tacticalRiskQuery += ` AND DATE(tactical_risk.review_date) = '${date}'`;
  }
  if (id) {
    tacticalRiskQuery += ` AND tactical_risk.id = '${id}'`;
  }
  let [tacticalRisk] = await db.query(tacticalRiskQuery);

  tacticalRisk = await decodeAndParseFields(tacticalRisk);
  return tacticalRisk;
}

export async function riskRegisterReviewDateAndApproval(
  templateFileUrl,
  templateName,
  subject,
  id
) {
  try {
    let date = id ? null : moment().subtract(2, "days").format("YYYY-MM-DD");

    const strategicRisk = await getStrategicRisk(date, id);
    const tacticalRisk = await getTacticalRisk(date, id);
    const companyWebsite = process.env.HARMONY_FRONTEND_URL;

    const allRisks = [...strategicRisk, ...tacticalRisk];

    for (let risk of allRisks) {
      const sendRecordArray = {
        name: `${risk.approver_name} ${risk.approver_surname}`,
        templateFileUrl: templateFileUrl
          ? templateFileUrl
          : "mail_for_strategic_tactical_register_review_date_approaching_template.html",
        templateName: templateName
          ? templateName
          : "Strategic Risk/ Tactical Risk Register Review Date",
        company_name: risk.organization_name,
        company_website: companyWebsite,
        risk_register_name: risk.risk_register_name,
        days: "2",
        risk_assessment_date: moment(risk.assessment_date).format("DD-MM-YYYY"),
        status: risk.status,
        business_structure: risk.organization_name,
        risk_review_date: moment(risk.review_date).format("DD-MM-YYYY"),
        risk_assessment_facilitator: `${risk.facilitator_name} ${risk.facilitator_surname}`,
        risk_register_approver: `${risk.approver_name} ${risk.approver_surname}`,
        risk_meeting_name: risk.meeting,
      };

      await sendDynamicEmail({
        to: risk.approver_email,
        // to: "mohammad.tariq.sartia@gmail.com",
        subject: subject
          ? subject
          : `Upcoming Risk Review Date - ${risk.risk_register_name}`,
        data: sendRecordArray,
      });
    }
  } catch (error) {
    throw error;
  }
}

export async function riskRegisterTemplateHelper(
  templateFileUrl,
  templateName,
  subject,
  id,
  strategic = true,
  to,
  additionalData = {}
) {
  try {
    const data = strategic
      ? await getStrategicRisk(null, id)
      : await getTacticalRisk(null, id);
    const companyWebsite = process.env.HARMONY_FRONTEND_URL;

    const risk = data[0];
    const sendRecordArray = {
      name: `${risk.approver_name} ${risk.approver_surname}`,
      templateFileUrl: templateFileUrl,
      templateName: templateName,
      company_name: risk.organization_name,
      company_website: companyWebsite,
      risk_register_name: risk.risk_register_name,
      risk_assessment_date: moment(risk.assessment_date).format("DD-MM-YYYY"),
      status: risk.status,
      business_structure: risk.organization_name,
      risk_review_date: moment(risk.review_date).format("DD-MM-YYYY"),
      risk_assessment_facilitator: `${risk.facilitator_name} ${risk.facilitator_surname}`,
      risk_register_approver: `${risk.approver_name} ${risk.approver_surname}`,
      risk_meeting_name: risk.meeting,
      ...additionalData,
    };

    await sendDynamicEmail({
      to: to ? to : risk.approver_email,
      // to: "mohammad.tariq.sartia@gmail.com",
      subject: subject,
      data: sendRecordArray,
    });
  } catch (error) {
    throw error;
  }
}

export async function getOperationalRiskData(id, date = null) {
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = operational_risk.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = operational_risk.organization",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = operational_risk.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as facilitator",
      onCondition:
        "facilitator.id = operational_risk.risk_assessment_facilitator",
    },
    {
      type: "left",
      targetTable: "users as approver",
      onCondition: "approver.id = operational_risk.risk_register_approver",
    },
    {
      type: "left",
      targetTable: "main_process",
      onCondition: "main_process.id = operational_risk.main_process",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  let operationalRiskQuery = `SELECT operational_risk.*, 
    main_process.name as main_process_name,
    organization.name as organization_name, users.name as created_by, users.surname as created_by_surname, users.email as created_by_email,
    facilitator.name as facilitator_name, facilitator.surname as facilitator_surname,facilitator.email as facilitator_email,
    approver.name as approver_name, approver.surname as approver_surname, approver.email as approver_email,
    meeting.meeting_title as meeting_name
    FROM operational_risk 
    ${joinsRecord} WHERE operational_risk.deleted = 0`;

  if (date) {
    operationalRiskQuery += ` AND DATE(operational_risk.review_date) = '${date}'`;
  }
  if (id) {
    operationalRiskQuery += ` AND operational_risk.id = '${id}'`;
  }
  let [operationalRisk] = await db.query(operationalRiskQuery);

  operationalRisk = await decodeAndParseFields(operationalRisk);

  // Prepare the structured data
  for (let risk of operationalRisk) {
    // Fetch associated work areas
    const [workAreas] = await db.query(
      `SELECT * FROM operational_work_area WHERE operational_risk_id = ${risk.id} AND deleted = 0`
    );
    risk.work_areas = await Promise.all(
      workAreas.map(async (workArea) => {
        const [activities] = await db.query(
          `SELECT * FROM operational_activities WHERE work_id = ${workArea.id} AND deleted = 0`
        );
        workArea.activities = await Promise.all(
          activities.map(async (activity) => {
            const [risks] = await db.query(
              `SELECT operational_risk_identification.*, category.name as category_name FROM operational_risk_identification  LEFT JOIN category ON category.id = operational_risk_identification.category WHERE operational_risk_identification.activity_id = ${activity.id} AND operational_risk_identification.deleted = 0`
            );
            activity.risks = risks.map((riskData) => ({
              ...riskData,
              control_assessments: JSON.parse(
                riskData.control_assessments || "[]"
              ),
              opportunity_description: JSON.parse(
                riskData.opportunity_description || "[]"
              ),
            }));
            return {
              ...activity,
              risks: await decodeAndParseFields(activity.risks),
            };
          })
        );
        return {
          ...workArea,
          activities: await decodeAndParseFields(workArea.activities),
        };
      })
    );
  }
  return operationalRisk;
}

export function generateOperationalRiskTable(data) {
  let tableRows = data.work_areas
    .map((workArea) =>
      workArea.activities
        .map((activity) =>
          activity.risks
            .map(
              (risk) => `
        <tr>
            <td>${data.organization_name}</td>
            <td>${workArea.name}</td>
            <td>${activity.name}</td>
            <td>${activity.frequency}</td>
            <td>${risk.name || "N/A"}</td>
            <td>${risk.category_name || "N/A"}</td>
            <td>${risk.description || "N/A"}</td>
        </tr>
      `
            )
            .join("")
        )
        .join("")
    )
    .join("");

  return `
    <div class="table-container">
        <table>
            <tr>
                <th>Business Structure</th>
                <th>Work Area</th>
                <th>Activity</th>
                <th>Activity Frequency</th>
                <th>Risk Name</th>
                <th>Risk Category</th>
                <th>Risk Description</th>
            </tr>
            ${tableRows}
        </table>
    </div>
  `;
}

export async function getOperationalRecordArray({
  operationalData,
  templateFileUrl,
  templateName,
}) {
  try {
    const companyWebsite = process.env.HARMONY_FRONTEND_URL;
    const sendRecordArray = {
      templateFileUrl: templateFileUrl,
      templateName: templateName,
      company_name: operationalData.organization_name,
      company_website: companyWebsite,
      risk_register_name: operationalData.risk_register_name,
      risk_assessment_date: moment(operationalData.assessment_date).format(
        "DD-MM-YYYY"
      ),
      status: operationalData.status,
      business_structure: operationalData.organization_name,
      risk_review_date: moment(operationalData.review_date).format(
        "DD-MM-YYYY"
      ),
      risk_assessment_facilitator: `${operationalData.facilitator_name} ${operationalData.facilitator_surname}`,
      risk_register_approver: `${operationalData.approver_name} ${operationalData.approver_surname}`,
      risk_meeting_name: operationalData.meeting_name,
      main_process: operationalData.main_process_name,
    };
    return sendRecordArray;

    // await sendDynamicEmail({
    //   to: to ? to : risk.approver_email,
    //   // to: "mohammad.tariq.sartia@gmail.com",
    //   subject: subject,
    //   data: sendRecordArray,
    // });
  } catch (error) {
    throw error;
  }
}

export async function operationalRiskReviewDate() {
  const twoDaysAgo = moment().subtract(2, "days").format("YYYY-MM-DD");

  const [operationalData] = await getOperationalRiskData(null, twoDaysAgo);

  const dataToSend = await getOperationalRecordArray({
    operationalData,
    templateFileUrl: "mail_for_operational_risk_review_date_template.html",
    templateName: "Operational Risk Register Review Date",
  });

  dataToSend.name = dataToSend.risk_register_approver;
  dataToSend.days = 2;
  sendDynamicEmail({
    to: operationalData.approver_email,
    subject: `Upcoming Risk Review Date - ${dataToSend.risk_register_name}`,
    data: dataToSend,
  });
}
function getArticle(word) {
  return /^[aeiouAEIOU]/.test(word) ? "An" : "A";
}

export function getTemplateName(schedule_type) {
  if (!schedule_type.includes("-")) {
    return (
      schedule_type.charAt(0).toUpperCase() +
      schedule_type.slice(1) +
      " Scheduling"
    );
  } else {
    return "Testing and Examination Scheduling";
  }
}

export async function getScheduleById(id) {
  const joins = [
    {
      type: "left",
      targetTable: "audit_type",
      onCondition: "audit_type.id = audit_scheduling.type",
    },
    {
      type: "left",
      targetTable: "users as owner",
      onCondition: "owner.id = audit_scheduling.owner",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = audit_scheduling.organization",
    },
    {
      type: "left",
      targetTable: "audit_scheduling AS reference",
      onCondition: "reference.id = audit_scheduling.reference_number",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = audit_scheduling.location",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  const auditSchedulingDataFetchQuery = `SELECT audit_scheduling.*, audit_type.name AS type_name , reference.name AS reference_number_name , organization.name as  organization_name, location.name as location_name, CONCAT(owner.name , ' ' , owner.surname) AS owner_name, owner.email as owner_email FROM audit_scheduling  ${joinsRecord} WHERE audit_scheduling.deleted = 0 AND audit_scheduling.id = ${id}`;

  let [auditSchedulingDataFetch] = await db.query(
    auditSchedulingDataFetchQuery
  );

  auditSchedulingDataFetch = await decodeAndParseFields(
    auditSchedulingDataFetch
  );

  for (let audit of auditSchedulingDataFetch) {
    // for specific record get name
    const linked_to = audit.linked_to;
    const record = audit.record_name;
    if (linked_to && record) {
      const tableName = auditLinkedTable[linked_to];
      const mappedFields = await buildSelectQuery(tableName);
      let query = `SELECT id, ${mappedFields} FROM ${tableName} WHERE deleted = 0 AND id = ${record}`;
      if (tableName === "permit_license_compliance") {
        query = `SELECT permit_license_compliance.id, permit_license.name as name FROM ${tableName} LEFT JOIN permit_license ON permit_license.id = permit_license_compliance.name WHERE permit_license_compliance.deleted = 0  AND permit_license_compliance.id = ${record}`;
      }
      const [result] = await db.query(query);
      audit.record_name_name = result[0]?.name;
    }

    const participants = audit.participants;

    let partDetails = [];
    if (participants) {
      [partDetails] = await db.query(
        `SELECT CONCAT(name , ' '  , surname ) AS name , profile, email FROM users WHERE id IN (${participants})`
      );
    }
    audit.participant_details = partDetails;
    let leadPersonDetails = [];
    const source = audit.source;
    if (audit?.lead_person?.length) {
      const lead_persons = audit.lead_person;
      if (source == "Internal") {
        if (Array.isArray(lead_persons) && lead_persons.length > 0) {
          // Filter out null or invalid values
          const validLeadPersons = lead_persons.filter(
            (id) => id !== null && id !== undefined && id !== ""
          );

          if (validLeadPersons.length > 0) {
            // Construct the query
            const leadIds = validLeadPersons.join(",");

            // Fetch data from the database
            [leadPersonDetails] = await db.query(
              `SELECT CONCAT(name, ' ', surname) AS name, profile, email 
                   FROM users 
                   WHERE id IN (${leadIds})`
            );
          }
        }

        // [leadPersonDetails] = await db.query(
        //   `SELECT CONCAT(name , ' '  , surname ) AS name , profile FROM users WHERE id IN (${lead_persons})`
        // );
        if (audit.schedule_type === "certifications") {
          audit.lead_person = leadPersonDetails;
        } else {
          audit.lead_person = leadPersonDetails;
        }
      } else {
        const lead_persons = audit.lead_person;
        [leadPersonDetails] = await db.query(
          `SELECT * FROM contractor_registration WHERE id IN (${lead_persons})`
        );
        for (let leadPerson of leadPersonDetails) {
          leadPerson.lead_person_name = leadPerson.contractor_name;
        }
        audit.lead_person = leadPersonDetails;
      }
    }
    const members = audit.members;
    if (Array.isArray(members) && members.length > 0) {
      if (source == "Internal") {
        const [membersList] = await db.query(
          `SELECT CONCAT(users.name , ' ' , users.surname) AS name , users.profile AS profile, users.email FROM users WHERE users.id IN (${members}) `
        );
        audit.member_details = membersList;
      } else {
        const [membersList] = await db.query(
          `SELECT contractor_name as name,email FROM contractor_registration WHERE id IN (${members}) `
        );
        audit.member_details = membersList;
      }
    }
    const criticalBusiness = audit.critical_business;
    if (Array.isArray(criticalBusiness) && criticalBusiness.length > 0) {
      const [criticalBusinessList] = await db.query(
        `SELECT id,name FROM business_processes WHERE id IN (${criticalBusiness})`
      );
      audit.critical_business_details = criticalBusinessList;
    }
  }
  return auditSchedulingDataFetch;
}

export function getSchedulingRecordArray({ schedulingData, templateName }) {
  const companyWebsite = process.env.HARMONY_FRONTEND_URL;
  const schedule_type = schedulingData.schedule_type;

  const article = getArticle(schedule_type);
  const schedule_type_with_article = `${article} ${schedule_type}`;

  const leadPersons = schedulingData.lead_person
    .map((person) => person.name)
    .join(", ");
  const auditMembers =
    schedulingData.members &&
    schedulingData.member_details.map((member) => member.name).join(", ");
  let participants;
  if (schedulingData.participant_type == "Internal") {
    participants = schedulingData.participant_details
      .map((participant) => participant.name)
      .join(", ");
  } else {
    participants = schedulingData.stakeholder
      .map((participant) => participant.stakeholder_name)
      .join(", ");
  }

  const criticalBusiness =
    schedulingData.critical_business &&
    schedulingData.critical_business_details
      .map((criticalBusiness) => criticalBusiness.name)
      .join(", ");
  const sendRecordArray = {
    templateFileUrl: "mail_for_scheduling_template.html",
    templateName: templateName,
    company_website: companyWebsite,
    company_name: schedulingData.organization_name,
    schedule_type_with_article,
    business_structure: schedulingData.organization_name,
    schedule_name: schedulingData.name,
    schedule_type,
    reference_number: schedulingData.reference_number_name,
    linked_to: schedulingData.linked_to,
    specific_record: schedulingData.record_name_name,
    start_date: moment(schedulingData.planned_start_date).format("DD-MM-YYYY"),
    end_date: moment(schedulingData.planned_end_date).format("DD-MM-YYYY"),
    want_to_repeat: schedulingData.want_to_repeat,
    repeat_when: schedulingData.repeat_when,
    cycles: schedulingData.cycles,
    source: schedulingData.source,
    scheduler_name: leadPersons,
    audit_members: auditMembers,
    owner: schedulingData.owner_name,
    stakeholders_involved: schedulingData.participant_type,
    participants,
    scope: schedulingData.scope,
    critical_business_processes: criticalBusiness,
    location: schedulingData?.location_name ?? "",
  };
  return sendRecordArray;
}

export async function ddrmExpiryDateApproaching() {
  const today = moment().format("YYYY-MM-DD");

  // Check if today is Monday for weekly reminders
  const isMonday = moment().isoWeekday() === 1;
  const joins = [
    {
      type: "left",
      targetTable: "document_type",
      onCondition: "document_type.id = document_creation.document_type",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = document_creation.organization",
    },
    {
      type: "left",
      targetTable: "users as owner",
      onCondition: "owner.id = document_creation.owner_user_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  const query = `SELECT document_creation.* , document_type.name AS document_type_name,organization.name AS organization_name,CONCAT(owner.name, ' ', owner.surname) AS owner_name, owner.email as owner_email, DATEDIFF(document_creation.expiry_date, ?) AS days_left
 FROM document_creation ${joinsRecord} WHERE document_creation.deleted = "0" AND (
      DATEDIFF(document_creation.expiry_date, ?) = 90 
      OR DATEDIFF(document_creation.expiry_date, ?) = 60
      OR DATEDIFF(document_creation.expiry_date, ?) = 30
      OR (DATEDIFF(document_creation.expiry_date, ?) > 0 AND ? = 1)  -- Weekly on Monday
    )`;

  let [result] = await db.query(query, [
    today,
    today,
    today,
    today,
    today,
    isMonday,
  ]);

  result = await decodeAndParseFields(result);

  const companyWebsite = process.env.HARMONY_FRONTEND_URL;

  for (let document of result) {
    let document_amendment =
      document.document_amendment == "1" ? "Revised" : "New";

    document.viewing_rights_details = await getUserListByIds(
      document?.viewing_rights_employees
    );
    document.modification_rights_details = await getUserListByIds(
      document?.modification_rights_employees
    );
    document.deleting_rights_details = await getUserListByIds(
      document?.deleting_rights_employees
    );

    let document_owner_name;
    let document_owner_email;
    if (document.document_owner === "External") {
      document_owner_name = document.specify_owner;
      document_owner_email = document.specify_owner_email;
    } else {
      document_owner_name = document.owner_name;
      document_owner_email = document.owner_email;
    }
    const recordDoc = {
      document_owner_name,
      document_owner: document.document_owner,
      document_name: document.document_name,
      templateFileUrl: "mail_for_document_expiry_date_template.html",
      templateName: "Document Expiry date reminder",
      company_website: companyWebsite,
      document_amendment,
      days: document.days_left,
      company_name: document.organization_name,
      business_structure: document.organization_name,
      document_type: document.document_type_name,
      document_description: document.description,
      document_version_number: document.version_number,
      document_hierarchy: document.document_hierarchy,
      document_classification: document.document_classification,
      document_origin: document.document_origin,
      viewing_rights_employee_name: document.viewing_rights_details
        .map((user) => user.name)
        .join(", "),
      modification_rights_employee_name: document.modification_rights_details
        .map((user) => user.name)
        .join(", "),
      deleting_rights_employee_name: document.deleting_rights_details
        .map((user) => user.name)
        .join(", "),
      document_effective_date: moment(document.effective_date).format(
        "DD-MMM-YYYY"
      ),
      document_expiry_date: moment(document.expiry_date).format("DD-MMM-YYYY"),
      document_status: document.document_status,
    };

    sendDynamicEmail({
      to: document_owner_email,
      subject: `Document Expiry Date Approaching - ${document.document_name}`,
      data: recordDoc,
    });
  }
}

export async function addUserLimitCheck() {
  const clientInternalId = process.env.CLIENT_INTERNAL_ID;
  const { data } = await axios.get(
    `${superadminApiUrl}/api/harmony-access/client-deployment/${clientInternalId}?includes=plan`
  );
  const { clientData, deploymentData } = data.data;
  const { plan, planId, maxUsers } = deploymentData;
  const users = await getRecord("users", "deleted", "0");
  const userCount = users.length;
  console.log("userCount: ", userCount);
  if (planId && plan) {
    const { name, maxUsers } = plan;
    if (userCount >= maxUsers) {
      console.log("User limit reached");
      throw new Error("User limit reached");
    } else {
      console.log("User limit not reached", "maxUsers", maxUsers);
      return true;
    }
  } else {
    console.log(
      "Plan not found means plan is custom all plan related data inside deployment"
    );
    const { maxUsers } = deploymentData;
    if (userCount >= maxUsers) {
      console.log("User limit reached");
      throw new Error("User limit reached");
    } else {
      console.log("User limit not reached", "maxUsers", maxUsers);
      return true;
    }
  }
}

export async function paginateRecords(records, queryParams) {
  let { page = 1, pageSize = 10, all = true } = queryParams;
  page = parseInt(page, 10);
  pageSize = parseInt(pageSize, 10);
  all = all === "false" || all === false;
  if (all) {
    const startIndex = (page - 1) * pageSize;
    const paginatedData = records.slice(startIndex, startIndex + pageSize);
    return {
      totalRecords: records.length,
      data: paginatedData,
    };
  } else {
    return { totalRecords: records.length, data: records };
  }
}

// Helper function for generating a random 6-digit hex color
function generateHexColor() {
  // Generate a random integer between 0 and FFFFFF (16777215)
  let randomColor = Math.floor(Math.random() * 16777216).toString(16);

  // Pad the string with leading zeros if it's shorter than 6 characters
  let hexColor = randomColor.padStart(6, '0');

  return "#" + hexColor;
}

// --- Modified getColorForCategory function ---
export async function getColorForCategory(categoryId, categoryType) {
  try {
    const checkColorQuery = `
      SELECT hex_color
      FROM color
      WHERE category_id = ? AND category_type = ?;
    `;
    const [existingColorResult] = await db.query(checkColorQuery, [
      categoryId,
      categoryType,
    ]);

    if (existingColorResult && existingColorResult.length > 0) {
      let hexColor = existingColorResult[0].hex_color;

      // --- Correction Logic Added Here ---
      // Check if the retrieved color string is shorter than the expected 7 characters (#RRGGBB)
      // and ensure it actually starts with '#' to avoid malformed data issues.
      if (hexColor && hexColor.startsWith('#') && hexColor.length < 7) {
        console.warn(`Retrieved short color '${hexColor}' for category ${categoryId} (${categoryType}). Correcting.`);
        // Extract the hex part (everything after '#')
        const hexPart = hexColor.substring(1);
        // Pad the hex part with leading zeros to ensure 6 digits
        const paddedHexPart = hexPart.padStart(6, '0');
        // Reconstruct the full, corrected hex color string
        hexColor = "#" + paddedHexPart;

        // --- Update the database with the corrected value ---        
        try {
          const updateColorQuery = `
            UPDATE color SET hex_color = ?
            WHERE category_id = ? AND category_type = ?;
          `;
          await db.query(updateColorQuery, [hexColor, categoryId, categoryType]);
          console.log(`Updated database with corrected color '${hexColor}' for category ${categoryId} (${categoryType}).`);
        } catch (updateError) {
          console.error(`Failed to update corrected color in database:`, updateError);
          // Continue execution even if update fails, returning the corrected color anyway.
        }
        
        // --- End Database Update ---

      }
      // --- End Correction Logic ---

      // Return the original color (if valid) or the corrected color
      return hexColor;

    } else {
      // No existing color found, generate a new, correctly formatted one
      const newHexColor = generateHexColor(); // Uses the fixed generator
      const insertColorQuery = `
        INSERT INTO color (category_id, category_type, hex_color)
        VALUES (?, ?, ?);
      `;
      await db.query(insertColorQuery, [categoryId, categoryType, newHexColor]);
      return newHexColor;
    }
  } catch (error) {
    console.error(`Error getting/creating color for category ${categoryId} (${categoryType}):`, error);
    // Optionally throw the error or return a default color
    return "#cccccc"; // Default grey color in case of error
  }
}


export async function updateRecord(tableName, where, data) {
  const setClause = Object.entries(data)
    .map(
      ([key, value]) =>
        `${key} = ${typeof value === "string" ? `'${value}'` : value}`
    )
    .join(", ");

  const whereClause = Object.entries(where)
    .map(
      ([key, value]) =>
        `${key} = ${typeof value === "string" ? `'${value}'` : value}`
    )
    .join(" AND ");

  const q = `UPDATE ${tableName} SET ${setClause} WHERE ${whereClause}`;
  // return console.log("q: ", q);
  const [result] = await db.query(q);
  return result;
}

export async function overdueAndPendingAction() {
  try {
    const today = moment().format("YYYY-MM-DD");

    const [pendingUpdate] = await db.query(`
      UPDATE custom_action_creation
      SET status = 'Pending'
      WHERE status = 'To-Do' 
      AND start_date < '${today}' 
      AND deleted = "0"
    `);

    const [overdueUpdate] = await db.query(`
      UPDATE custom_action_creation
      SET status = 'Over-Due'
      WHERE status <> 'Completed' 
      AND end_date < '${today}' 
      AND deleted = "0"
    `);
    console.log("Updated pending records:", pendingUpdate.affectedRows);
    console.log("Updated overdue records:", overdueUpdate.affectedRows);
  } catch (error) {
    throw error;
  }
}

export async function createUnifiedUserApi(userId) {
  try {
    let data = {};
    // get user details using its id and also format domain for deployment detail checks

    const domain = extractSubdomain(process.env.HARMONY_FRONTEND_URL);
    console.log("domain: ", domain);
    data.domain = domain;

    const userQuery = `SELECT id , name , email ,  surname , profile , phone  FROM users WHERE id = '${userId}'`;
    const [userData] = await db.query(userQuery);
    if (userData && userData.length > 0) {
      data.name = userData[0]?.name;
      data.email = userData[0]?.email;
      data.surname = userData[0]?.surname;
      data.profile = userData[0]?.profile;
      data.phone = userData[0]?.phone;
    }
    const response = await axios.post(
      `${process.env.SUPER_ADMIN_BACKEND_URL}/api/auth/create-unified-user`,
      data
    );
    const unifiedUserId = response.data.data.uniqueId;
    const updateUserQuery = `UPDATE users SET unified_user_uique_id = '${unifiedUserId}' WHERE id = '${userId}'`;
    const [updateUser] = await db.query(updateUserQuery);
    return true;
  } catch (error) {
    console.log("Error from unified user creation", error);
    return false;
  }
}

export function extractSubdomain(url) {
  const match = url.match(/^https:\/\/([^\.]+)/);
  return match ? match[1] : null;
}

export async function checkUnifiedUserApi(data) {
  try {
    const isLive = extractSubdomain(process.env.HARMONY_FRONTEND_URL);
    if (!isLive) {
      return true;
    }
    const response = await axios.post(
      `${process.env.SUPER_ADMIN_BACKEND_URL}/api/auth/unified-user-check`,
      data
    );
  } catch (err) {
    console.log("Error from unified user check", err);
    throw err;
  }
}
export function getPathAfterHost(inputUrl) {
  const parsedUrl = new URL(inputUrl);
  return parsedUrl.pathname
}

export function getDomain(url) {
  let domain = new URL(url).hostname;
  return domain;
}