import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createUpdateActivity,
  getActionActivity,
} from "../controller/actionActivitiesController.js";

/**
 * @swagger
 * tags:
 *   name: Action Activity
 *   description: API operations for managing activity
 */

const actionActivityRoutes = Router();

/**
 * @swagger
 * /api/actionActivity/create-update-action-activity:
 *   post:
 *     tags: [Action Activity]
 *     summary: Create or update an action activity
 *     description: Creates or update a new action activity
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *           example:
 *             custom_action_id : "name"
 *             type : "start/end"
 *     responses:
 *       200:
 *         description: Action Activity created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

actionActivityRoutes.post(
  "/create-update-action-activity",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      233,
      "action_template_list"
    )(req, res, next);
  },
  controllerWrapper(createUpdateActivity)
);

/**
 * @swagger
 * /api/actionActivity/get-action-activity:
 *   get:
 *     tags: [Action Activity]
 *     summary: Get a list of action activity
 *     description: Retrieves a list of all action activity.
 *     responses:
 *       200:
 *         description: A list of action activity.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *       401:
 *         description: Unauthorized.
 */

actionActivityRoutes.get(
  "/get-action-activity",
  verifyToken,
  sideBarPermissionCheck("view", 233, "action_template_list"),
  controllerWrapper(getActionActivity)
);

/**
 * @swagger
 * /api/actionActivity/get-action-activity/{id}:
 *   get:
 *     tags: [Action Activity]
 *     summary: Get a specific action activity
 *     description: Retrieves a list of all action activity.
 *     responses:
 *       200:
 *         description: A list of action activity.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *       401:
 *         description: Unauthorized.
 */

actionActivityRoutes.get(
  "/get-action-activity/:id",
  verifyToken,
  sideBarPermissionCheck("view", 233, "action_template_list"),
  controllerWrapper(getActionActivity)
);

export default actionActivityRoutes;
