import { Router } from "express";
import {
  createUpdateActionComments,
  getComments,
  deleteComments,
} from "../controller/actionCommentsController.js";

import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Action Comments
 *   description: API operations for managing comments
 */

const actionCommentsRoutes = Router();

/**
 * @swagger
 * /api/actionComments/create-update-comments:
 *   post:
 *     tags: [Action Comments]
 *     summary: Create or update an action comments
 *     description: Creates or update a new action comment
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *           example:
 *             id: 1
 *             custom_action_id : "name"
 *             comments : "comments"
 *     responses:
 *       200:
 *         description: Action Comments created or updated successfully.
 *       400:
 *         description: Bad request.
 *       401:
 *         description: Unauthorized.
 */

actionCommentsRoutes.post(
  "/create-update-comments",
  verifyToken,
  controllerWrapper(createUpdateActionComments)
);

/**
 * @swagger
 * /api/actionComments/get-comments:
 *   get:
 *     tags: [Action Comments]
 *     summary: Get a list of action comments
 *     description: Retrieves a list of all action comments.
 *     responses:
 *       200:
 *         description: A list of action comments.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 type: object
 *       401:
 *         description: Unauthorized.
 */

actionCommentsRoutes.get(
  "/get-comments",
  verifyToken,
  controllerWrapper(getComments)
);

/**
 * @swagger
 * /api/actionComments/get-comments/{id}:
 *   get:
 *     tags: [Action Comments]
 *     summary: Get an action comments by ID
 *     description: Retrieves an action comments by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the action comments to retrieve.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: The action comments details.
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Action Comments not found.
 */

actionCommentsRoutes.get(
  "/get-comments/:id",
  verifyToken,
  controllerWrapper(getComments)
);

/**
 * @swagger
 * /api/actionComments/delete-comments/{id}:
 *   delete:
 *     tags: [Action Comments]
 *     summary: Delete an action comments by ID
 *     description: Deletes an action comments by its ID.
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: The ID of the action comments to delete.
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Action Comments deleted successfully.
 *       401:
 *         description: Unauthorized.
 *       404:
 *         description: Action Comments not found.
 */

actionCommentsRoutes.delete(
  "/delete-comments/:id",
  verifyToken,
  controllerWrapper(deleteComments)
);

export default actionCommentsRoutes;
