import { Router } from "express";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { createUpdateBcpTesting, deleteBcpTesting, getBcpTesting } from "../controller/bcpTestingController.js";

const BcpTestingRoutes = Router();
/**
 * @swagger
 * tags:
 *   name: BCP Testing
 *   description: API endpoints for managing BCP Testing
 */

/**
 * @swagger
 * /api/bcpTesting/create-update-bcp-testing:
 *   post:
 *     summary: Create or update a BCP Testing entry
 *     tags:
 *       - BCP Testing
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: ID of the BCP Testing entry (required for update)
 *               planned_testing_date:
 *                 type: string
 *                 format: date
 *               plan_name:
 *                 type: string
 *               actual_testing_date:
 *                 type: string
 *                 format: date
 *               location:
 *                 type: integer
 *               method:
 *                 type: string
 *               objective:
 *                 type: string
 *               text_coordinator:
 *                 type: string
 *               team_members:
 *                 type: string
 *               external_participants:
 *                 type: string
 *               test_scenario_description:
 *                 type: string
 *               assumptions:
 *                 type: string
 *               critical_function_tested:
 *                 type: string
 *               recovery_objectives:
 *                 type: string
 *               test_outcomes:
 *                 type: string
 *               issues_identified:
 *                 type: string
 *               recovery_times:
 *                 type: string
 *               plan_deviations:
 *                 type: string
 *               improvements_identified:
 *                 type: string
 *               action_items:
 *                 type: string
 *               key_learnings:
 *                 type: string
 *               team_feedback:
 *                 type: string
 *               document_attachment:
 *                 type: string
 *               reviewer:
 *                 type: string
 *             example:
 *               planned_testing_date: "2024-12-01"
 *               plan_name: "Annual Disaster Recovery Test"
 *               actual_testing_date: "2024-12-02"
 *               location: 101
 *               method: "Simulation"
 *               objective: "Validate disaster recovery procedures"
 *               text_coordinator: "John Doe"
 *               team_members: "Team A, Team B"
 *               external_participants: "Vendor A"
 *               test_scenario_description: "Simulating a power outage scenario"
 *               assumptions: "All systems are operational before testing"
 *               critical_function_tested: "Database failover"
 *               recovery_objectives: "Restore critical services within 2 hours"
 *               test_outcomes: "Successful failover with minor delays"
 *               issues_identified: "Network lag in failover process"
 *               recovery_times: "1.5 hours"
 *               plan_deviations: "Deviated due to incomplete documentation"
 *               improvements_identified: "Update recovery procedures"
 *               action_items: "Update failover process documentation"
 *               key_learnings: "Better pre-test preparation needed"
 *               team_feedback: "Test was well-coordinated"
 *               document_attachment: "link-to-document.pdf"
 *               reviewer: "Jane Smith"
 *     responses:
 *       '200':
 *         description: Successfully created or updated the BCP Testing entry
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */

/**
 * @swagger
 * /api/bcpTesting/get-bcp-testing:
 *   get:
 *     summary: Get all BCP Testing entries
 *     tags:
 *       - BCP Testing
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       '200':
 *         description: Successfully retrieved all BCP Testing entries
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */

/**
 * @swagger
 * /api/bcpTesting/get-bcp-testing/{id}:
 *   get:
 *     summary: Get a BCP Testing entry by ID
 *     tags:
 *       - BCP Testing
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the BCP Testing entry
 *     responses:
 *       '200':
 *         description: Successfully retrieved the BCP Testing entry
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: BCP Testing entry not found
 *       '500':
 *         description: Server error
 */

/**
 * @swagger
 * /api/bcpTesting/delete-bcp-requirement-analysis/{id}:
 *   delete:
 *     summary: Delete a BCP Testing entry by ID
 *     tags:
 *       - BCP Testing
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the BCP Testing entry
 *     responses:
 *       '200':
 *         description: Successfully deleted the BCP Testing entry
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: BCP Testing entry not found
 *       '500':
 *         description: Server error
 */

BcpTestingRoutes.post(
  "/create-update-bcp-testing",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 213, "bcp_testing")(
      req,
      res,
      next
    );
  },
  controllerWrapper(createUpdateBcpTesting)
);

BcpTestingRoutes.get(
    "/get-bcp-testing",
    verifyToken,
    sideBarPermissionCheck("view", 213, "bcp_testing"),
    controllerWrapper(getBcpTesting)
  );

  BcpTestingRoutes.get(
    "/get-bcp-testing/:id",
    verifyToken,
    sideBarPermissionCheck("view", 213, "bcp_testing"),
    controllerWrapper(getBcpTesting)
  );

  BcpTestingRoutes.delete(
    "/delete-bcp-requirement-analysis/:id",
    verifyToken,
    sideBarPermissionCheck("delete", 213, "bcp_testing"),
    controllerWrapper(deleteBcpTesting)
  );

export default BcpTestingRoutes;
