import { Router } from "express";
import {
  createUpdateCompetencyAssessment,
  getCompetencyAssessment,
  deleteCompetencyAssessment,
} from "../controller/competencyAssessmentController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const competencyAssessmentRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Competency Assessment
 *   description: API endpoints for managing competency assessments
 */

/**
 * @swagger
 * /api/competencyAssessment/create-update-competency-assessment:
 *   post:
 *     summary: Create or update a competency assessment
 *     tags:
 *       - Competency Assessment
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               surname:
 *                 type: string
 *               organization:
 *                 type: integer
 *               department:
 *                 type: integer
 *               employee_id:
 *                 type: string
 *               gender:
 *                 type: string
 *               disability:
 *                 type: string
 *               race:
 *                 type: string
 *               employee_type:
 *                 type: string
 *               role_name:
 *                 type: string
 *               assessment_id:
 *                 type: string
 *               assessment_name:
 *                 type: string
 *               assessment_type:
 *                 type: string
 *               assessment_description:
 *                 type: string
 *               job_role:
 *                 type: integer
 *               competencies_assessed:
 *                 type: string
 *               assessor_date:
 *                 type: string
 *                 format: date
 *               assessor_name:
 *                 type: integer
 *               assessor_method:
 *                 type: string
 *               assessor_duration:
 *                 type: string
 *                 format: time
 *               assessment_location:
 *                 type: string
 *               assessment_result:
 *                 type: string
 *               competency_level:
 *                 type: string
 *               follow_up_actions:
 *                 type: string
 *           example:
 *             name: John
 *             surname: Doe
 *             organization: 1
 *             department: 2
 *             employee_id: "EMP001"
 *             gender: Male
 *             disability: "None"
 *             race: "Asian"
 *             employee_type: "Full-time"
 *             role_name: "Developer"
 *             assessment_id: "ASSESS001"
 *             assessment_name: "Technical Skills Assessment"
 *             assessment_type: "Performance Review"
 *             assessment_description: "Assessment of technical skills and performance"
 *             job_role: 3
 *             competencies_assessed: "Coding skills, problem-solving"
 *             assessor_date: "2024-07-20"
 *             assessor_name: 4
 *             assessor_method: "Interview"
 *             assessor_duration: "01:30:00"
 *             assessment_location: "Office"
 *             assessment_result: "Passed"
 *             competency_level: "Advanced"
 *             follow_up_actions: "None required"
 *     responses:
 *       '200':
 *         description: Competency assessment created or updated successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
competencyAssessmentRoutes.post(
  "/create-update-competency-assessment",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      213,
      "competency_assessment"
    )(req, res, next);
  },
  controllerWrapper(createUpdateCompetencyAssessment)
);

/**
 * @swagger
 * /api/competencyAssessment/get-competency-assessment:
 *   get:
 *     summary: Get all competency assessments
 *     tags:
 *       - Competency Assessment
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       '200':
 *         description: Successfully retrieved competency assessments
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
competencyAssessmentRoutes.get(
  "/get-competency-assessment",
  verifyToken,
  sideBarPermissionCheck("view", 213, "competency_assessment"),
  controllerWrapper(getCompetencyAssessment)
);

/**
 * @swagger
 * /api/competencyAssessment/get-competency-assessment/{id}:
 *   get:
 *     summary: Get a competency assessment by ID
 *     tags:
 *       - Competency Assessment
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: string
 *         description: ID of the competency assessment to retrieve
 *     responses:
 *       '200':
 *         description: Successfully retrieved the competency assessment
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: Competency assessment not found
 *       '500':
 *         description: Server error
 */
competencyAssessmentRoutes.get(
  "/get-competency-assessment/:id",
  verifyToken,
  sideBarPermissionCheck("view", 213, "competency_assessment"),
  controllerWrapper(getCompetencyAssessment)
);

/**
 * @swagger
 * /api/competencyAssessment/delete-competency-assessment:
 *   delete:
 *     summary: Delete a competency assessment
 *     tags:
 *       - Competency Assessment
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       '200':
 *         description: Competency assessment deleted successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
competencyAssessmentRoutes.delete(
  "/delete-competency-assessment/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 213, "competency_assessment"),
  controllerWrapper(deleteCompetencyAssessment)
);

export default competencyAssessmentRoutes;
