import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorSafeWorking,
  getContractorSafeWorking,
  deleteContractorSafeWorking,
  sendSafeWorking,
  fillForm,
  uploadDocument,
} from "../controller/contractorSafeWorkingController.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Contractor SafeWorking
 *   description: API operations for managing contractor safe workings
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     ContractorSafeWorking:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the contractor safe working
 *         contractor:
 *           type: integer
 *           description: ID of the contractor
 *         status:
 *           type: string
 *           description: The current status of the safe working
 *         form:
 *           type: string
 *           description: The form data in long text format
 *         public_link:
 *           type: string
 *           description: Public link for the safe working
 *         organization:
 *           type: integer
 *           description: ID of the associated organization
 *         department:
 *           type: integer
 *           description: ID of the associated department
 *         deleted:
 *           type: integer
 *           description: Whether the safe working is deleted (0 or 1)
 *         created_by:
 *           type: integer
 *           description: ID of the user who created the safe working
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the safe working was created
 *         updated_by:
 *           type: integer
 *           description: ID of the user who last updated the safe working
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The date and time when the safe working was last updated
 */

const contractorSafeWorkingRoutes = Router();

/**
 * @swagger
 * /api/contractorSafeWorking/create-update-contractor-safe-working:
 *   post:
 *     summary: Create or update a contractor safe working
 *     tags: [Contractor SafeWorking]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorSafeWorking'
 *     responses:
 *       200:
 *         description: The contractor safe working was successfully created/updated
 */

contractorSafeWorkingRoutes.post(
  "/create-update-contractor-safe-working",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 272, "contractor_safe_working")(req, res, next);
  },
  controllerWrapper(createUpdateContractorSafeWorking)
);

/**
 * @swagger
 * /api/contractorSafeWorking/get-contractor-safe-working:
 *   get:
 *     summary: Get all contractor safe workings
 *     tags: [Contractor SafeWorking]
 *     responses:
 *       200:
 *         description: List of all contractor safe workings
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/ContractorSafeWorking'
 */

contractorSafeWorkingRoutes.get(
  "/get-contractor-safe-working",
  verifyToken,
  sideBarPermissionCheck("view", 272, "contractor_safe_working"),
  controllerWrapper(getContractorSafeWorking)
);

/**
 * @swagger
 * /api/contractorSafeWorking/get-contractor-safe-working/{id}:
 *   get:
 *     summary: Get a contractor safe working by ID
 *     tags: [Contractor SafeWorking]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor safe working
 *     responses:
 *       200:
 *         description: Contractor safe working data
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/ContractorSafeWorking'
 */

contractorSafeWorkingRoutes.get(
  "/get-contractor-safe-working/:id",
  verifyToken,
  sideBarPermissionCheck("view", 272, "contractor_safe_working"),
  controllerWrapper(getContractorSafeWorking)
);

/**
 * @swagger
 * /api/contractorSafeWorking/delete-contractor-safe-working/{id}:
 *   delete:
 *     summary: Delete a contractor safe working by ID
 *     tags: [Contractor SafeWorking]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor safe working to delete
 *     responses:
 *       200:
 *         description: Contractor safe working was deleted
 */

contractorSafeWorkingRoutes.delete(
  "/delete-contractor-safe-working/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 272, "contractor_safe_working"),
  controllerWrapper(deleteContractorSafeWorking)
);

/**
 * @swagger
 * /api/contractorSafeWorking/send-safe-working:
 *   get:
 *     summary: Send a contractor safe working
 *     tags: [Contractor SafeWorking]
 *     responses:
 *       200:
 *         description: The safe working was successfully sent
 */

contractorSafeWorkingRoutes.post(
  "/send-safe-working",
  verifyToken,
  sideBarPermissionCheck("view", 272, "contractor_safe_working"),
  controllerWrapper(sendSafeWorking)
);

/**
 * @swagger
 * /api/contractorSafeWorking/fill-safe-working:
 *   post:
 *     summary: Fill out a contractor safe working form
 *     tags: [Contractor SafeWorking]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/ContractorSafeWorking'
 *     responses:
 *       200:
 *         description: The form was successfully submitted
 */

contractorSafeWorkingRoutes.post("/fill-safe-working", controllerWrapper(fillForm));

/**
 * @swagger
 * /api/contractorSafeWorking/upload-safe-working:
 *   post:
 *     summary: Upload safe working document
 *     tags: [Contractor SafeWorking]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id : 1
 *             document : file
 *     responses:
 *       200:
 *         description: Document uploaded successfully
 */

contractorSafeWorkingRoutes.post(
  "/upload-safe-working",
  verifyToken,
  sideBarPermissionCheck("edit", 272, "contractor_safe_working"),
  controllerWrapper(uploadDocument)
);
export default contractorSafeWorkingRoutes;
