// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateDepartment,
  deleteDepartment,
  getParentDepartmentList,
  importDepartmentData,
  listOfEmployeeAccordingToDepartment,
  viewDepartment,
} from "../controller/departmentController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { departmentCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Department
 *   description: API operations for managing all departments
 */
const departmentRoutes = Router();

/**
 * @swagger
 * /api/departments/create-update-department:
 *   post:
 *     summary: Create or update a department
 *     tags: [Department]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               head:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example: {
 *             id: "id",
 *             name: "name",
 *             description: "description",
 *             head: "head",
 *             organization: []
 *           }
 */
departmentRoutes.post(
  "/create-update-department",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 17, "department")(
      req,
      res,
      next
    );
  },
  validateRequest(departmentCreateUpdateValidationSchema),
  controllerWrapper(createUpdateDepartment)
);

/**
 * @swagger
 * /api/departments/get-departments:
 *   get:
 *     summary: Get a list of all departments
 *     tags: [Department]
 */
departmentRoutes.get(
  "/get-departments",
  verifyToken,
  sideBarPermissionCheck("view", 17, "department"),
  controllerWrapper(viewDepartment)
);

/**
 * @swagger
 * /api/departments/get-departments/{id}:
 *   get:
 *     summary: Get a specific record of department
 *     tags: [Department]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the department to retrieve
 *         schema:
 *           type: string
 */
departmentRoutes.get(
  "/get-departments/:id",
  verifyToken,
  sideBarPermissionCheck("view", 17, "department"),
  controllerWrapper(viewDepartment)
);

/**
 * @swagger
 * /api/departments/parent-department-list:
 *   get:
 *     summary: Get parent department list
 *     tags: [Department]
 *     parameters:
 *       - name: organizations
 *         in: query
 *         required: true
 *         description: IDs of the organizations
 *         schema:
 *           type: string
 */

departmentRoutes.get(
  "/parent-department-list",
  verifyToken,
  sideBarPermissionCheck("view", 17, "department"),
  controllerWrapper(getParentDepartmentList)
);


/**
 * @swagger
 * /api/departments/delete-departments/{id}:
 *   delete:
 *     summary: Delete a specific record of department
 *     tags: [Department]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the department to delete
 *         schema:
 *           type: string
 */
departmentRoutes.delete(
  "/delete-departments/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 17, "department"),
  controllerWrapper(deleteDepartment)
);

/**
 * @swagger
 * /api/departments/department-employee-list:
 *   get:
 *     summary: Get a list of employees according to department
 *     tags: [Department]
 */
departmentRoutes.get(
  "/department-employee-list",
  verifyToken,
  sideBarPermissionCheck("view", 17, "department"),
  controllerWrapper(listOfEmployeeAccordingToDepartment)
);

/**
 * @swagger
 * /api/departments/import-department-data:
 *   post:
 *     summary: Import records of departments
 *     tags: [Department]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               data:
 *                 type: array
 *                 items:
 *                   type: object
 *                   properties:
 *                     name:
 *                       type: string
 *                     description:
 *                       type: string
 *                     head:
 *                       type: string
 *                     organization:
 *                       type: array
 *                       items:
 *                         type: string
 *           example: {
 *             data: [
 *               {
 *                 name: "Department Name",
 *                 description: "Department Description",
 *                 head: "Head Name",
 *                 organization: []
 *               }
 *             ]
 *           }
 */
departmentRoutes.post(
  "/import-department-data",
  verifyToken,
  sideBarPermissionCheck("create", 17, "department", "organization"),
  controllerWrapper(importDepartmentData)
);

export default departmentRoutes;
