import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createEnvironmentalIncidentsForm,
  deleteEnvironmentalIncidentsForm,
  getEnvironmentalIncidentsForm,
} from "../controller/environmentalIncidentFormController.js";

const environmentalIncidentsFromRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Environmental Incident Form
 *   description: API operations for managing environmental incident forms
 */

/**
 * @swagger
 * /api/environmentalIncidentsForm/create-environmental-incidents-form:
 *   post:
 *     summary: Create a new environmental incident form
 *     tags: [Environmental Incident Form]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               category_id:
 *                 type: integer
 *               incident_id:
 *                 type: integer
 *               date_of_incident:
 *                 type: string
 *                 format: date
 *               incident_ref_no:
 *                 type: string
 *               spillage_or_discharge_type:
 *                 type: string
 *               pollutant_type:
 *                 type: string
 *               quantity_litres:
 *                 type: string
 *               source_of_discharge:
 *                 type: string
 *               site_area_affected_spillage:
 *                 type: string
 *               size_of_area_affected:
 *                 type: string
 *               proximity_to_watercourses:
 *                 type: string
 *               leakage_type:
 *                 type: string
 *               source_of_leak:
 *                 type: string
 *               quantity_estimated_leak:
 *                 type: string
 *               site_area_affected_leakage:
 *                 type: string
 *               emission_type:
 *                 type: string
 *               source_of_emission:
 *                 type: string
 *               color_of_emission:
 *                 type: string
 *               duration_of_pollution:
 *                 type: string
 *               illegal_dumping_type:
 *                 type: string
 *               waste_material_classification:
 *                 type: string
 *               quantity_kg_litres:
 *                 type: string
 *               source_origin_dumping:
 *                 type: string
 *               site_area_affected_dumping:
 *                 type: string
 *               unauthorized_activity_name:
 *                 type: string
 *               commencement_date_activity:
 *                 type: string
 *                 format: date
 *               area_affected_activity:
 *                 type: string
 *               breach_type:
 *                 type: string
 *               legislative_requirement_name:
 *                 type: string
 *               reason_for_non_compliance:
 *                 type: string
 *               potential_fine_or_penalty:
 *                 type: string
 *               duration_of_pollution_breach:
 *                 type: string
 *               case_reported_to_regulator:
 *                 type: string
 *               comments:
 *                 type: string
 *               form_completed_by:
 *                 type: string
 *               designation:
 *                 type: string
 *               organization:
 *                 type: integer
 *               department:
 *                 type: integer
 *           example: {
 *             category_id: 1,
 *             incident_id: 123,
 *             date_of_incident: "2023-12-31",
 *             incident_ref_no: "REF123",
 *             spillage_or_discharge_type: "Spillage",
 *             pollutant_type: "Chemical",
 *             quantity_litres: "100",
 *             source_of_discharge: "Factory",
 *             site_area_affected_spillage: "Warehouse",
 *             size_of_area_affected: "500 sq ft",
 *             proximity_to_watercourses: "Nearby river",
 *             leakage_type: "Gas",
 *             source_of_leak: "Pipeline",
 *             quantity_estimated_leak: "50",
 *             site_area_affected_leakage: "Factory area",
 *             emission_type: "Smoke",
 *             source_of_emission: "Chimney",
 *             color_of_emission: "Black",
 *             duration_of_pollution: "1 hour",
 *             illegal_dumping_type: "Waste",
 *             waste_material_classification: "Hazardous",
 *             quantity_kg_litres: "200",
 *             source_origin_dumping: "Unknown",
 *             site_area_affected_dumping: "Dumping site",
 *             unauthorized_activity_name: "Unknown activity",
 *             commencement_date_activity: "2023-01-01",
 *             area_affected_activity: "Site area",
 *             breach_type: "Regulatory",
 *             legislative_requirement_name: "Environmental Act",
 *             reason_for_non_compliance: "Negligence",
 *             potential_fine_or_penalty: "5000",
 *             duration_of_pollution_breach: "1 week",
 *             case_reported_to_regulator: "Yes",
 *             comments: "Initial report",
 *             form_completed_by: "John Doe",
 *             designation: "Manager",
 *             organization: 1,
 *             department: 2
 *           }
 *     responses:
 *       200:
 *         description: Environmental incident form successfully created
 */
environmentalIncidentsFromRoutes.post(
  "/create-environmental-incidents-form",
  verifyToken,
  controllerWrapper(createEnvironmentalIncidentsForm)
);

/**
 * @swagger
 * /api/environmentalIncidentsForm/get-environmental-incidents-form/{id}:
 *   get:
 *     summary: Get a specific environmental incident form by ID
 *     tags: [Environmental Incident Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the environmental incident form
 *     responses:
 *       200:
 *         description: Environmental incident form data
 */
environmentalIncidentsFromRoutes.get(
  "/get-environmental-incidents-form/:id",
  verifyToken,
  controllerWrapper(getEnvironmentalIncidentsForm)
);

/**
 * @swagger
 * /api/environmentalIncidentsForm/delete-environmental-incidents-form/{id}:
 *   delete:
 *     summary: Delete a specific environmental incident form by ID
 *     tags: [Environmental Incident Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the environmental incident form to delete
 *     responses:
 *       200:
 *         description: Environmental incident form successfully deleted
 */
environmentalIncidentsFromRoutes.delete(
  "/delete-environmental-incidents-form/:id",
  verifyToken,
  controllerWrapper(deleteEnvironmentalIncidentsForm)
);

export default environmentalIncidentsFromRoutes;
