import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateIncidentCategory,
  deleteIncidentCategory,
  viewIncidentCategory,
} from "../controller/incidentCategoryController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { validateRequest } from "../helper/general.js";
import { incidentCategoryCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
import { controllerWrapper } from "../helper/wrapper.js";

const incidentCategoryRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Incident Category
 *   description: API operations for managing all Incident Categories
 */

/**
 * @swagger
 * /api/incidentCategory/create-update-incident-category:
 *   post:
 *     summary: Create or update an Incident Category
 *     tags: [Incident Category]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: integer
 *           example:
 *             id: 1
 *             name: "Safety Incident"
 *             description: "Category for safety-related incidents"
 *             organization: [1, 2]
 */
incidentCategoryRoutes.post(
  "/create-update-incident-category",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      175,
      "incident_category"
    )(req, res, next);
  },
  validateRequest(incidentCategoryCreateUpdateValidationSchema),
  controllerWrapper(createUpdateIncidentCategory)
);

/**
 * @swagger
 * /api/incidentCategory/get-incident-category:
 *   get:
 *     summary: Get list of all Incident Categories
 *     tags: [Incident Category]
 */
incidentCategoryRoutes.get(
  "/get-incident-category",
  verifyToken,
  // sideBarPermissionCheck("view", 175, "incident_category"),
  controllerWrapper(viewIncidentCategory)
);

/**
 * @swagger
 * /api/incidentCategory/get-incident-category/{id}:
 *   get:
 *     summary: Get a specific record of an Incident Category
 *     tags: [Incident Category]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Incident Category to retrieve
 */
incidentCategoryRoutes.get(
  "/get-incident-category/:id",
  verifyToken,
  // sideBarPermissionCheck("view", 175, "incident_category"),
  controllerWrapper(viewIncidentCategory)
);

/**
 * @swagger
 * /api/incidentCategory/delete-incident-category/{id}:
 *   delete:
 *     summary: Delete a specific record of an Incident Category
 *     tags: [Incident Category]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the Incident Category to delete
 */
incidentCategoryRoutes.delete(
  "/delete-incident-category/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 175, "incident_category"),
  controllerWrapper(deleteIncidentCategory)
);

export default incidentCategoryRoutes;
