// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedIncident,
  deleteIncident,
  viewAllIncident,
  viewIncidentAuditTrail,
  viewIncidentInvestigation,
} from "../controller/incidentController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { incidentCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
import { validateRequest } from "../helper/general.js";

/**
 * @swagger
 * tags:
 *   name: Incident
 *   description: API operations for managing all MeetingType
 */
const incidentRoutes = Router();

/**
 * @swagger
 * api/incident/create-update-incident:
 *   post:
 *     summary: Create or update a Incident
 *     tags: [Incident]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             {
 *               id: 1,
 *             }
 */

incidentRoutes.post(
  "/create-update-incident",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 173, "incident")(
      req,
      res,
      next
    );
  },
  validateRequest(incidentCreateUpdateValidationSchema, false),
  controllerWrapper(createUpdatedIncident)
);

/**
 * @swagger
 * api/incident/get-incident:
 *   get:
 *     summary: Get all list of Incident
 *     tags: [Incident]
 */
incidentRoutes.get(
  "/get-incident",
  verifyToken,
  sideBarPermissionCheck("view", 173, "incident"),
  controllerWrapper(viewAllIncident)
);

/**
 * @swagger
 * get-incident-audit-trial/:id:
 *   get:
 *     summary: Get all list of Incident
 *     tags: [Incident]
 */
incidentRoutes.get(
  "/get-incident-audit-trial/:id",
  verifyToken,
  sideBarPermissionCheck("view", 173, "incident"),
  controllerWrapper(viewIncidentAuditTrail)
);

/**
 * @swagger
 * api/incident/get-incident-category/:incidentId:
 *   get:
 *     summary: Get all list of Incident Category whose form is related to incident
 *     tags: [Incident]
 */
incidentRoutes.get(
  "/get-incident-category/:incidentId",
  verifyToken,
  controllerWrapper(viewIncidentInvestigation)
);

/**
 * @swagger
 * api/incident/get-incident/{id}:
 *   get:
 *     summary: Get a specific record of Incident
 *     tags: [Incident]
 */
incidentRoutes.get(
  "/get-incident/:id",
  verifyToken,
  sideBarPermissionCheck("view", 173, "incident"),
  controllerWrapper(viewAllIncident)
);

/**
 * @swagger
 * api/incident/delete-incident/{id}:
 *   delete:
 *     summary: delete a specific record of area Type
 *     tags: [Incident]
 */
incidentRoutes.delete(
  "/delete-incident/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 173, "incident"),
  controllerWrapper(deleteIncident)
);

export default incidentRoutes;
