import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

import {
  createInjuryOnDutyForm,
  deleteInjuryOnDutyForm,
  getInjuryOnDutyForm,
} from "../controller/injuryOnDutyFormController.js";

const injuryOnDutyFromRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Injury On Duty Form
 *   description: API operations for managing injury on duty forms
 */

/**
 * @swagger
 * /api/injuryOnDutyForm/create-injury-on-duty-form:
 *   post:
 *     summary: Create a new injury on duty form
 *     tags: [Injury On Duty Form]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               incident_id:
 *                 type: integer
 *               category_id:
 *                 type: integer
 *               date_of_incident:
 *                 type: string
 *                 format: date
 *               incident_ref_no:
 *                 type: string
 *               name_of_injured:
 *                 type: string
 *               id_number:
 *                 type: string
 *               employee_type:
 *                 type: string
 *               experience_years:
 *                 type: integer
 *               experience_months:
 *                 type: integer
 *               role_on_the_incident:
 *                 type: integer
 *               licensed_trained:
 *                 type: string
 *               alcohol_substance_testing_done:
 *                 type: string
 *               tested_by:
 *                 type: string
 *               state_name:
 *                 type: string
 *               alcohol_substance_test_results:
 *                 type: string
 *               method_of_testing:
 *                 type: string
 *               type_of_incident_category:
 *                 type: string
 *               absence_start_date:
 *                 type: string
 *                 format: date
 *               absence_end_date:
 *                 type: string
 *                 format: date
 *               injury_level:
 *                 type: string
 *               name_of_clinic:
 *                 type: string
 *               name_of_occupational_nurse:
 *                 type: string
 *               name_of_hospital:
 *                 type: string
 *               name_of_doctor_practice_no:
 *                 type: string
 *               contact_details_treatment:
 *                 type: string
 *               type_of_injury:
 *                 type: string
 *               parts_of_body_affected:
 *                 type: string
 *               ppe_worn:
 *                 type: string
 *               type_of_ppe_worn:
 *                 type: string
 *               reportable_cases_classification:
 *                 type: string
 *               reported_to_authority_regulator:
 *                 type: string
 *               name_of_regulator:
 *                 type: string
 *               date_reported:
 *                 type: string
 *                 format: date
 *               reported_by_who:
 *                 type: string
 *               reported_to_saps:
 *                 type: string
 *               name_of_police_station:
 *                 type: string
 *               case_no:
 *                 type: string
 *               name_of_contact_person:
 *                 type: string
 *               contact_details_saps:
 *                 type: string
 *               estimated_cost:
 *                 type: number
 *                 format: float
 *               first_aid_applied:
 *                 type: string
 *               details_of_first_aider:
 *                 type: string
 *               comments:
 *                 type: string
 *               form_completed_by:
 *                 type: string
 *               designation:
 *                 type: string
 *               organization:
 *                 type: integer
 *               department:
 *                 type: integer
 *           example:
 *             incident_id: 123
 *             category_id: 1
 *             date_of_incident: "2023-12-31"
 *             incident_ref_no: "INC123"
 *             name_of_injured: "John Doe"
 *             id_number: "ID123"
 *             employee_type: "Full-time"
 *             experience_years: 5
 *             experience_months: 3
 *             role_on_the_incident: 2
 *             licensed_trained: "Yes"
 *             alcohol_substance_testing_done: "No"
 *             tested_by: "Internal"
 *             state_name: "California"
 *             alcohol_substance_test_results: "Negative"
 *             method_of_testing: "Breathalyzer"
 *             type_of_incident_category: "Workplace injury"
 *             absence_start_date: "2023-01-01"
 *             absence_end_date: "2023-01-07"
 *             injury_level: "Moderate"
 *             name_of_clinic: "Health Clinic"
 *             name_of_occupational_nurse: "Jane Smith"
 *             name_of_hospital: "City Hospital"
 *             name_of_doctor_practice_no: "Dr. Smith"
 *             contact_details_treatment: "123-456-7890"
 *             type_of_injury: "Fracture"
 *             parts_of_body_affected: "Leg"
 *             ppe_worn: "Yes"
 *             type_of_ppe_worn: "Hard hat"
 *             reportable_cases_classification: "Major"
 *             reported_to_authority_regulator: "Yes"
 *             name_of_regulator: "Health Department"
 *             date_reported: "2023-01-02"
 *             reported_by_who: "Manager"
 *             reported_to_saps: "No"
 *             name_of_police_station: "N/A"
 *             case_no: "N/A"
 *             name_of_contact_person: "N/A"
 *             contact_details_saps: "N/A"
 *             estimated_cost: 1000.00
 *             first_aid_applied: "Yes"
 *             details_of_first_aider: "Provided first aid at the scene."
 *             comments: "Incident report details."
 *             form_completed_by: "Manager"
 *             designation: "Manager"
 *             organization: 1
 *             department: 2
 *     responses:
 *       200:
 *         description: Injury on duty form successfully created
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
injuryOnDutyFromRoutes.post(
  "/create-injury-on-duty-form",
  verifyToken,
  // sideBarPermissionCheck(
  //   req.body.id ? "edit" : "create",
  //   176,
  //   "safety_incident_technical_form"
  // ),
  controllerWrapper(createInjuryOnDutyForm)
);

/**
 * @swagger
 * /api/injuryOnDutyForm/get-injury-on-duty-form/{id}:
 *   get:
 *     summary: Get a specific injury on duty form by ID
 *     tags: [Injury On Duty Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the injury on duty form
 *     responses:
 *       200:
 *         description: Injury on duty form data
 *       404:
 *         description: Injury on duty form not found
 *       500:
 *         description: Internal server error
 */
injuryOnDutyFromRoutes.get(
  "/get-injury-on-duty-form/:id",
  verifyToken,
  // sideBarPermissionCheck("view", 176, "safety_incident_technical_form"),
  controllerWrapper(getInjuryOnDutyForm)
);

/**
 * @swagger
 * /api/injuryOnDutyForm/delete-injury-on-duty-form/{id}:
 *   delete:
 *     summary: Delete a specific injury on duty form by ID
 *     tags: [Injury On Duty Form]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the injury on duty form to delete
 *     responses:
 *       200:
 *         description: Injury on duty form successfully deleted
 *       404:
 *         description: Injury on duty form not found
 *       500:
 *         description: Internal server error
 */
injuryOnDutyFromRoutes.delete(
  "/delete-injury-on-duty-form/:id",
  verifyToken,
  // sideBarPermissionCheck("delete", 176, "safety_incident_technical_form"),
  controllerWrapper(deleteInjuryOnDutyForm)
);

export default injuryOnDutyFromRoutes;
