// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedMeasurementMetric,
  deleteMeasurementMetric,
  viewAllMeasurementMetric,
} from "../controller/measurementMetricController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { measurementMetricCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
import { validateRequest } from "../helper/general.js";

/**
 * @swagger
 * tags:
 *   name: Measurement Metric
 *   description: API operations for managing all Measurement Metric
 */
const measurementMetricRoutes = Router();

/**
 * @swagger
 * /api/measurementMetric/create-update-measurement-metric:
 *   post:
 *     summary: Create or Update a Measurement Metric
 *     tags: [Measurement Metric]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             name: "Metric Name"
 *             description: "Metric Description"
 *             organization: [1, 2]
 *     responses:
 *       200:
 *         description: Measurement Metric created or updated successfully
 *       400:
 *         description: Bad request
 *       500:
 *         description: Internal server error
 */
measurementMetricRoutes.post(
  "/create-update-measurement-metric",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      139,
      "measurement_metric"
    )(req, res, next);
  },
  validateRequest(measurementMetricCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedMeasurementMetric)
);

/**
 * @swagger
 * /api/measurementMetric/get-measurement-metric:
 *   get:
 *     summary: Get list of all Measurement Metrics
 *     tags: [Measurement Metric]
 *     responses:
 *       200:
 *         description: A list of measurement metrics
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
measurementMetricRoutes.get(
  "/get-measurement-metric",
  verifyToken,
  sideBarPermissionCheck("view", 139, "measurement_metric"),
  controllerWrapper(viewAllMeasurementMetric)
);

/**
 * @swagger
 * /api/measurementMetric/get-measurement-metric/{id}:
 *   get:
 *     summary: Get a specific record of Measurement Metric
 *     tags: [Measurement Metric]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the measurement metric
 *     responses:
 *       200:
 *         description: A specific measurement metric record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
measurementMetricRoutes.get(
  "/get-measurement-metric/:id",
  verifyToken,
  sideBarPermissionCheck("view", 139, "measurement_metric"),
  controllerWrapper(viewAllMeasurementMetric)
);

/**
 * @swagger
 * /api/measurementMetric/delete-measurement-metric/{id}:
 *   delete:
 *     summary: Delete a specific record of Measurement Metric
 *     tags: [Measurement Metric]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The ID of the measurement metric to delete
 *     responses:
 *       200:
 *         description: Measurement Metric deleted successfully
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
measurementMetricRoutes.delete(
  "/delete-measurement-metric/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 139, "measurement_metric"),
  controllerWrapper(deleteMeasurementMetric)
);

export default measurementMetricRoutes;
