import { Router } from "express";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateNearMissReporting,
  getNearMissReporting,
  deleteNearMissReporting,
} from "../controller/nearMissReportingController.js";
import { verifyToken } from "../helper/tokenVerify.js";

const nearMissReportingRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Near Miss Reporting Forms
 *   description: API operations for managing near miss reporting forms
 */
/**
 * @swagger
 * /api/nearMissReporting/create-update-near-miss:
 *   post:
 *     summary: Create a new Near Miss Reporting Form
 *     tags: [Near Miss Reporting Forms]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: {
 *               "employee_id": 1,
 *               "near_miss_reporting_date": "2024-11-08",
 *               "near_miss_actual_date": "2024-11-07",
 *               "near_miss_classification": "Safety Hazard",
 *               "immediate_action_taken": "Isolated the area and informed safety officer.",
 *               "near_miss_description": "A loose wire was exposed in the workspace.",
 *               "severity": "Medium",
 *               "probability": "High",
 *               "requires_investigation": true,
 *               "investigation_outcome": "No injuries, but equipment damage was noted.",
 *               "immediate_cause": "Loose wiring during maintenance.",
 *               "root_cause": "Improper inspection post-maintenance.",
 *               "near_miss_submission": [
 *                 {
 *                   "user_id": 1,
 *                   "role": 1,
 *                   "department": 1
 *                 }
 *               ],
 *               "notification": [
 *                 {
 *                   "user_id": 1,
 *                   "role": 1,
 *                   "department": 1
 *                 }
 *               ],
 *               "organization": 1,
 *               "department": 2,
 *               "deleted": 0,
 *               "created_by": 1,
 *               "updated_by": 2
 *           }
 *     responses:
 *       200:
 *         description: Successfully created Near Miss Reporting Form
 *       400:
 *         description: Bad request
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */

nearMissReportingRoutes.post(
  "/create-update-near-miss",
  verifyToken,
  controllerWrapper(createUpdateNearMissReporting)
);


/**
 * @swagger
 * /api/nearMissReporting/get-near-miss:
 *   get:
 *     summary: Retrieve  Near Miss Reporting Form 
 *     tags: [Near Miss Reporting Forms]
 */
nearMissReportingRoutes.get(
  "/get-near-miss",
  verifyToken,
  controllerWrapper(getNearMissReporting)
);


/**
 * @swagger
 * /api/nearMissReporting/get-near-miss/{id}:
 *   get:
 *     summary: Retrieve a Near Miss Reporting Form by ID
 *     tags: [Near Miss Reporting Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Successfully retrieved Near Miss Reporting Form
 *       404:
 *         description: Near Miss Reporting Form not found
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
nearMissReportingRoutes.get(
  "/get-near-miss/:id",
  verifyToken,
  controllerWrapper(getNearMissReporting)
);

/**
 * @swagger
 * /api/nearMissReporting/delete-near-miss/{id}:
 *   delete:
 *     summary: Delete a Near Miss Reporting Form by ID
 *     tags: [Near Miss Reporting Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Successfully deleted Near Miss Reporting Form
 *       404:
 *         description: Near Miss Reporting Form not found
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
nearMissReportingRoutes.delete(
  "/delete-near-miss/:id",
  verifyToken,
  controllerWrapper(deleteNearMissReporting)
);

export default nearMissReportingRoutes;
