import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateOrganization,
  deleteOrganization,
  getUsersOrganization,
} from "../controller/organizationController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { organizationCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Business Structure
 *   description: API operations for managing Business Structure
 */
const organizationRoutes = Router();

/**
 * @swagger
 * /api/organization/create-update-organization:
 *   post:
 *     summary: Create or update an organization
 *     tags: [Business Structure]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               name:
 *                 type: string
 *               logo:
 *                 type: string
 *               parent_id:
 *                 type: integer
 *               description:
 *                 type: string
 *           example:
 *             id: 1
 *             name: "Organization Name"
 *             logo: "logo_url"
 *             parent_id: 0
 *             description: "Description of the organization"
 */
organizationRoutes.post(
  "/create-update-organization",
  verifyToken,
  validateRequest(organizationCreateUpdateValidationSchema),
  controllerWrapper(createUpdateOrganization)
);

/**
 * @swagger
 * /api/organization/get-organization:
 *   get:
 *     summary: Get all organizations
 *     tags: [Business Structure]
 */
organizationRoutes.get(
  "/get-organization",
  verifyToken,
  controllerWrapper(getUsersOrganization)
);

/**
 * @swagger
 * /api/organization/get-organization/{id}:
 *   get:
 *     summary: Get a specific organization by ID
 *     tags: [Business Structure]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the organization
 */
organizationRoutes.get(
  "/get-organization/:id",
  verifyToken,
  controllerWrapper(getUsersOrganization)
);

/**
 * @swagger
 * /api/organization/delete-organization/{id}:
 *   delete:
 *     summary: Delete an organization by ID
 *     tags: [Business Structure]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the organization
 */
organizationRoutes.delete(
  "/delete-organization/:id",
  verifyToken,
  controllerWrapper(deleteOrganization)
);

export default organizationRoutes;
