// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedPermitLicense,
  deletePermitLicense,
  viewAllPermitLicense,
} from "../controller/permitLicenseComplianceController.js";
import { validateRequest } from "../helper/general.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { permitLicenseComplianceCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Permit License
 *   description: API operations for managing all Permit Licenses
 */
const permitLicenseComplianceRoutes = Router();

/**
 * @swagger
 * /api/permitLicense/create-update-permit-license:
 *   post:
 *     summary: Create or update a Permit License
 *     tags: [Permit License]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               title:
 *                 type: string
 *               description:
 *                 type: string
 *               permit_holder:
 *                 type: string
 *               permit_license_number:
 *                 type: string
 *               permit_type:
 *                 type: string
 *               issuing_authority:
 *                 type: string
 *               applicable_section_of_legislation:
 *                 type: string
 *               issue_date:
 *                 type: string
 *                 format: date
 *               validity_from:
 *                 type: string
 *                 format: date
 *               validity_until:
 *                 type: string
 *                 format: date
 *               notification_alert:
 *                 type: string
 *               condition_description:
 *                 type: string
 *               renewal_due_date:
 *                 type: string
 *                 format: date
 *               permit_renewal_process:
 *                 type: string
 *               department:
 *                 type: string
 *               organization:
 *                 type: string
 *               employee_name:
 *                 type: string
 *               role_name:
 *                 type: string
 *           example:
 *             id: 1234
 *             title: "Environmental Permit"
 *             description: "Permit for operating a waste management facility"
 *             permit_holder: "ABC Waste Management Inc."
 *             permit_license_number: "PERMIT-2022-123"
 *             permit_type: "Waste Management"
 *             issuing_authority: "Environmental Protection Agency"
 *             applicable_section_of_legislation: "Environmental Protection Act"
 *             issue_date: "2022-01-15"
 *             validity_from: "2022-02-01"
 *             validity_until: "2023-01-31"
 *             notification_alert: "30 days before expiry"
 *             condition_description: "Compliance with emission standards"
 *             renewal_due_date: "2022-12-01"
 *             permit_renewal_process: "Submit renewal application with updated documents"
 *             department: "Environmental Compliance Department"
 *             organization: "ABC Waste Management Inc."
 *             employee_name: "John Doe"
 *             role_name: "Environmental Compliance Officer"
 */
permitLicenseComplianceRoutes.post(
  "/create-update-permit-license-compliance",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      109,
      "permit_license_compliance"
    )(req, res, next);
  },
  validateRequest(permitLicenseComplianceCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedPermitLicense)
);

/**
 * @swagger
 * /api/permitLicense/get-permit-license:
 *   get:
 *     summary: Get all Permit Licenses
 *     tags: [Permit License]
 */
permitLicenseComplianceRoutes.get(
  "/get-permit-license-compliance",
  verifyToken,
  sideBarPermissionCheck("view", 109, "permit_license_compliance"),
  controllerWrapper(viewAllPermitLicense)
);

/**
 * @swagger
 * /api/permitLicense/get-permit-license/{id}:
 *   get:
 *     summary: Get a specific Permit License by ID
 *     tags: [Permit License]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the Permit License
 */
permitLicenseComplianceRoutes.get(
  "/get-permit-license-compliance/:id",
  verifyToken,
  sideBarPermissionCheck("view", 109, "permit_license_compliance"),
  controllerWrapper(viewAllPermitLicense)
);

/**
 * @swagger
 * /api/permitLicense/delete-permit-license/{id}:
 *   delete:
 *     summary: Delete a specific Permit License by ID
 *     tags: [Permit License]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the Permit License
 */
permitLicenseComplianceRoutes.delete(
  "/delete-permit-license-compliance/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 109, "permit_license_compliance"),
  controllerWrapper(deletePermitLicense)
);

export default permitLicenseComplianceRoutes;
