    // Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateRoleHierarchy,
  deleteRoleHierarchy,
  viewRoleHierarchy,
} from "../controller/roleHierarchyController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Role Hierarchy
 *   description: API operations for managing all Role Hierarchy
 */
const roleHierarchyRoutes = Router();

/**
 * @swagger
 * /api/roleHierarchy/create-update-role-hierarchy:
 *   post:
 *     summary: Create or update an Role Hierarchy
 *     tags: [Role Hierarchy]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the role-hierarchy"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Role Hierarchy created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
roleHierarchyRoutes.post(
  "/create-update-role-hierarchy",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 339, "role_hierarchy")(
      req,
      res,
      next
    );
  },
  controllerWrapper(createUpdateRoleHierarchy)
);

/**
 * @swagger
 * /api/roleHierarchy/get-role-hierarchy:
 *   get:
 *     summary: Get list of all RoleHierarchy
 *     tags: [Role Hierarchy]
 *     responses:
 *       200:
 *         description: List of roleHierarchy
 *       500:
 *         description: Internal server error
 */
roleHierarchyRoutes.get(
  "/get-role-hierarchy",
  verifyToken,
  sideBarPermissionCheck("view", 339, "role_hierarchy"),
  controllerWrapper(viewRoleHierarchy)
);

/**
 * @swagger
 * /api/roleHierarchy/get-role-hierarchy/{id}:
 *   get:
 *     summary: Get a specific record of Role Hierarchy
 *     tags: [Role Hierarchy]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the role-hierarchy
 *     responses:
 *       200:
 *         description: Role Hierarchy data
 *       404:
 *         description: Role Hierarchy not found
 *       500:
 *         description: Internal server error
 */
roleHierarchyRoutes.get(
  "/get-role-hierarchy/:id",
  verifyToken,
  sideBarPermissionCheck("view", 339, "role_hierarchy"),
  controllerWrapper(viewRoleHierarchy)
);

/**
 * @swagger
 * /api/roleHierarchy/delete-role-hierarchy/{id}:
 *   delete:
 *     summary: Delete a specific record of Role Hierarchy
 *     tags: [Role Hierarchy]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the role-hierarchy to delete
 *     responses:
 *       200:
 *         description: Role Hierarchy successfully deleted
 *       404:
 *         description: Role Hierarchy not found
 *       500:
 *         description: Internal server error
 */
roleHierarchyRoutes.delete(
  "/delete-role-hierarchy/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 339, "role_hierarchy"),
  controllerWrapper(deleteRoleHierarchy)
);

export default roleHierarchyRoutes; 
