// AssessmentSchema.js
import sequelize from "./sequelize.js"; // Adjust the import path as necessary
import Organization from "./OrganizationSchema.js";
import User from "./UserSchema.js";
import Department from "./DepartmentSchema.js";
import EmployeeType from "./EmployeeTypeSchema.js"; // Assuming an EmployeeType schema exists
import ContractorRegistration from "./ContractorRegistrationSchema.js"; // Assuming a Contractor schema exists
import Location from "./LocationSchema.js"; // Assuming a Location schema exists
import Currency from "./CurrencySchema.js";
import { DataTypes } from "sequelize";

const Assessment = sequelize.define(
  "Assessment",
  {
    id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    assessment_name: {
      type: DataTypes.STRING(250), // Assessment Name
      allowNull: true,
    },
    assessment_type: {
      type: DataTypes.INTEGER, // Dropdown for Assessment Type
      allowNull: true,
    },
    linked_to: {
      type: DataTypes.INTEGER, // Dropdown for Linked To
      allowNull: true,
    },
    specific_record_id: {
      type: DataTypes.INTEGER, // Dropdown for Specific Record, fetched from the relevant module
      allowNull: true,
    },
    planned_assessment_start_date: {
      type: DataTypes.DATE, // Planned Assessment Start Date
      allowNull: true,
    },
    planned_assessment_end_date: {
      type: DataTypes.DATE, // Planned Assessment End Date
      allowNull: true,
    },
    assessment_source: {
      type: DataTypes.ENUM("Internal", "External"), // Source can be Internal or External
      allowNull: true,
    },
    // Internal Assessor
    internal_assessor_id: {
      type: DataTypes.INTEGER, // Assessor Name fetched from Employee List
      allowNull: true,
      references: {
        model: EmployeeType,
        key: "id",
      },
    },
    // External Assessor
    external_assessor_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select for external assessors (fetched from contractors)
      allowNull: true,
      references: {
        model: ContractorRegistration,
        key: "id",
      },
    },
    assessment_owner: {
      type: DataTypes.STRING(250), // Assessment Owner
      allowNull: true,
    },
    // Stakeholders Involved
    stakeholders_involved: {
      type: DataTypes.ENUM("Internal", "External"),
      allowNull: true,
    },
    internal_stakeholder_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select for internal stakeholders
      allowNull: true,
      references: {
        model: EmployeeType,
        key: "id",
      },
    },
    external_stakeholder_name: {
      type: DataTypes.STRING(250), // Free Text for External Stakeholder Name
      allowNull: true,
    },
    external_stakeholder_email: {
      type: DataTypes.STRING(250), // Email Input for External Stakeholder
      allowNull: true,
      validate: {
        isEmail: true, // Validate email format
      },
    },
    assessment_location_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select Dropdown for Assessment Locations
      allowNull: true,
      references: {
        model: Location,
        key: "id",
      },
    },
    assessment_scope: {
      type: DataTypes.TEXT("long"), // Description Field for Assessment Scope
      allowNull: true,
    },
    critical_business_processes: {
      type: DataTypes.TEXT, // Critical Business Processes
      allowNull: true,
    },

    // Technological Resources
    technological_resources: {
      type: DataTypes.TEXT, // Add More field for Technological Resources
      allowNull: true,
    },

    // Financial Resources
    financial_resources: {
      type: DataTypes.ENUM("Financial Resource Type", "Value", "Currency"),
      allowNull: true,
    },
    financial_resource_type: {
      type: DataTypes.ENUM("Operating Budget", "Capital Budget"), // Enum for Financial Resource Type
      allowNull: true,
    },
    value: {
      type: DataTypes.FLOAT, // Numerical Input for Financial Value
      allowNull: true,
      validate: {
        isFloat: true, // Validate float input
        min: 0, // Minimum value constraint
      },
    },
    currency_id: {
      type: DataTypes.INTEGER, // Dropdown for Currency, referencing prefilled dataset
      allowNull: true,
      references: {
        model: Currency,
        key: "id",
      },
    },
    evaluation_tool: {
      type: DataTypes.ENUM("Applicable", "Not Applicable"), // Dropdown for Evaluation Tool
      allowNull: true,
    },
    department: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Department,
        key: "id",
      },
    },
    organization: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Organization,
        key: "id",
      },
    },
    created_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    created_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    deleted: {
      type: DataTypes.TINYINT,
      allowNull: true,
      defaultValue: 0,
    },
    updated_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
      onUpdate: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    updated_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
  },
  {
    tableName: "assessment",
    timestamps: false,
    charset: "utf8mb4",
    collate: "utf8mb4_general_ci",
  }
);

// Associations
Assessment.belongsTo(EmployeeType, {
  as: "internalAssessor",
  foreignKey: "internal_assessor_id",
});
Assessment.belongsToMany(ContractorRegistration, {
  as: "externalAssessors",
  through: "AssessmentExternalAssessors",
  foreignKey: "assessment_id",
});
Assessment.belongsToMany(EmployeeType, {
  as: "internalStakeholders",
  through: "AssessmentInternalStakeholders",
  foreignKey: "assessment_id",
});
Assessment.belongsToMany(Location, {
  as: "assessmentLocations",
  through: "AssessmentLocationMapping",
  foreignKey: "assessment_id",
});
Assessment.belongsTo(Currency, { as: "currency", foreignKey: "currency_id" });
Assessment.belongsTo(Organization, { as: "org", foreignKey: "organization" });
Assessment.belongsTo(User, { as: "creator", foreignKey: "created_by" });
Assessment.belongsTo(User, { as: "updater", foreignKey: "updated_by" });
Assessment.belongsTo(Department, { as: "dep", foreignKey: "department" });

export default Assessment;
