import archiver from "archiver";
import fs from "fs";
import { fileURLToPath } from "url";
import path from "path";
import { exec } from "child_process";
import dotenv from "dotenv";
dotenv.config(); // Load environment variables from .env file
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
// Get the filename from the command line arguments
const filename = process.argv[2] || "production";
const output = fs.createWriteStream(path.join(__dirname, `${filename}.zip`));
const archive = archiver("zip", {
  zlib: { level: 9 },
});
output.on("close", function () {
  console.log(archive.pointer() + " total bytes");
  console.log(`${filename}.zip has been created`);
});
archive.on("error", function (err) {
  throw err;
});
archive.pipe(output);
const exportDatabase = () => {
  return new Promise((resolve, reject) => {
    const host = process.env.DATABASE_HOST || "localhost";
    const user = process.env.DATABASE_USER || "root";
    const password = process.env.DATABASE_PASSWORD || "";
    const database = process.env.DATABASE_NAME || "harmony_help4";
    const mysqldumpPath =
      "C:\\Program Files\\MySQL\\MySQL Server 8.0\\bin\\mysqldump.exe";
    const outputPath = `${__dirname}/${database}.sql`;
    const dumpCommand = `"${mysqldumpPath}" -u ${user} -p${password} -h ${host} ${database} > "${outputPath}"`;
    console.log("Executing command:", dumpCommand); // Log the command for debugging
    exec(dumpCommand, (error, stdout, stderr) => {
      if (error) {
        reject(`Error exporting database: ${stderr}`);
      } else {
        resolve(outputPath);
      }
    });
  });
};

const fixNamedImports = () => {
  const projectDir = __dirname;

  const traverseDirectory = (dir) => {
    fs.readdirSync(dir).forEach((file) => {
      const filePath = path.join(dir, file);
      const stat = fs.statSync(filePath);

      if (shouldSkipFile(filePath)) {
        return;
      }

      if (stat.isDirectory()) {
        traverseDirectory(filePath);
      } else if (path.extname(filePath) === ".js") {
        fixNamedImportsInFile(filePath);
      }
    });
  };

  const shouldSkipFile = (filePath) => {
    const relativePath = path.relative(projectDir, filePath);
    return (
      relativePath.startsWith("node_modules") ||
      relativePath.startsWith(".git") ||
      relativePath === "Prod.zip" ||
      relativePath === "zipProduction.js"
    );
  };

  const fixNamedImportsInFile = (filePath) => {
    try {
      let fileContent = fs.readFileSync(filePath, "utf8");
      let hasChanges = false;

      const newContent = fileContent.replace(
        /import\s+(.+?)\s+from\s+['"](.+?)['"]/g,
        (match, importClause, modulePath) => {
          if (modulePath.startsWith(".") || modulePath.startsWith("..")) {
            const absoluteModulePath = path.resolve(
              path.dirname(filePath),
              modulePath
            );
            const moduleDir = path.dirname(absoluteModulePath);
            const moduleBasename = path.basename(modulePath);

            const files = fs.readdirSync(moduleDir);
            const actualFileName = files.find(
              (file) =>
                file.toLowerCase() === moduleBasename.toLowerCase() ||
                file.toLowerCase() === (moduleBasename + ".js").toLowerCase()
            );

            if (actualFileName && actualFileName !== moduleBasename) {
              // Keep the .js extension in the import statement
              const correctedModulePath = modulePath.replace(
                moduleBasename,
                actualFileName
              );
              hasChanges = true;
              console.log(
                `In ${filePath}, corrected import: ${modulePath} -> ${correctedModulePath}`
              );
              return `import ${importClause} from '${correctedModulePath}'`;
            }
          }
          return match;
        }
      );

      if (hasChanges) {
        fs.writeFileSync(filePath, newContent, "utf8");
        console.log(`Fixed imports in file: ${filePath}`);
      }
    } catch (error) {
      console.error(`Error fixing imports in file: ${filePath}`, error);
    }
  };

  traverseDirectory(projectDir);
};

// Exclude node_modules and zip everything else
const zipFiles = async () => {
  try {
    const sqlFile = await exportDatabase();
    fixNamedImports(); // Fix named imports before zipping

    archive.glob("**/*", {
      ignore: [
        "node_modules/**",
        `${filename}.zip`,
        "zipProduction.js",
        ".env",
        "db-config.js",
        "public/**",
      ],
    });
    archive.file(sqlFile, { name: path.basename(sqlFile) });
    archive.finalize();
  } catch (error) {
    console.error("Failed to export database:", error);
  }
};
zipFiles();
